<?php
// Page pour éditer une leçon existante.
// Cette page s'inspire de create_lesson.php mais pré‑remplit les champs
// avec les valeurs existantes et met à jour la base de données à la soumission.

require_once __DIR__ . '/functions.php';

// Assurer que l'utilisateur est connecté
if (!isLoggedIn()) {
    redirect('login.php');
}
$user = currentUser($pdo);

// Récupérer l'ID de la leçon à modifier
$lessonId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$lesson = getLessonById($pdo, $lessonId);
if (!$lesson) {
    redirect('dashboard.php');
}
$cycle = getCycleById($pdo, $lesson['cycle_id']);
if (!$cycle) {
    redirect('dashboard.php');
}
$class = getClassById($pdo, $cycle['class_id']);
// Vérifier que l'utilisateur a le droit de modifier cette classe (enseignant, coordinateur ou admin)
if (!$class || !userCanAccessClass($pdo, $cycle['class_id'], $user['id']) || (!userHasRole($pdo, 'teacher') && !userHasRole($pdo, 'coordinator') && !userHasRole($pdo, 'admin'))) {
    redirect('dashboard.php');
}

// Messages d'erreur ou de succès
$error = '';
$success = '';

// Au POST, mettre à jour les champs
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $date = $_POST['lesson_date'] ?? '';
    $title = trim($_POST['title'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $fps = trim($_POST['fps'] ?? '');
    $cp = trim($_POST['cp'] ?? '');
    if (!$date || !$title) {
        $error = "La date et le titre de la leçon sont obligatoires.";
    } else {
        // Mise à jour de la leçon existante
        $stmt = $pdo->prepare('UPDATE lessons SET lesson_date = ?, title = ?, description = ?, fps = ?, cp = ? WHERE id = ?');
        $result = $stmt->execute([$date, $title, $description, $fps, $cp, $lessonId]);
        if ($result) {
            $success = "Leçon mise à jour avec succès.";
            // Actualiser les valeurs locales pour que le formulaire reflète les modifications
            $lesson['lesson_date'] = $date;
            $lesson['title'] = $title;
            $lesson['description'] = $description;
            $lesson['fps'] = $fps;
            $lesson['cp'] = $cp;
        } else {
            $error = "Erreur lors de la mise à jour de la leçon.";
        }
    }
}

// Récupérer la liste des FPS pour le menu déroulant
$fpsList = getFPSLibrary($pdo);

$pageTitle = 'Modifier la leçon';
include 'header.php';
?>

<div class="mt-4">
    <h2>Éditer la leçon pour le cycle <?php echo htmlspecialchars($cycle['name']); ?></h2>

    <p><a href="lesson_detail.php?id=<?php echo $lessonId; ?>" class="btn btn-secondary btn-sm">← Retour aux détails de la leçon</a></p>

    <?php if ($error): ?>
        <div class="alert alert-danger mt-3"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    <?php if ($success): ?>
        <div class="alert alert-success mt-3"><?php echo htmlspecialchars($success); ?></div>
    <?php endif; ?>

    <form method="post" action="" class="mt-3">
        <div class="mb-3">
            <label for="lesson_date" class="form-label">
                Date de la leçon
                <i class="bi bi-question-circle-fill ms-1 text-secondary" role="button" tabindex="0" data-bs-toggle="popover" data-bs-trigger="focus" title="Date" data-bs-content="La date de la séance permet de situer et d’ordonner les leçons dans le cycle."></i>
            </label>
            <input type="date" id="lesson_date" name="lesson_date" class="form-control" value="<?php echo htmlspecialchars($lesson['lesson_date']); ?>" required>
        </div>
        <div class="mb-3">
            <label for="title" class="form-label">
                Titre de la leçon
                <i class="bi bi-question-circle-fill ms-1 text-secondary" role="button" tabindex="0" data-bs-toggle="popover" data-bs-trigger="focus" title="Titre" data-bs-content="Indiquez un titre court et explicite pour la leçon (ex : ‘Appuis renversés’)."></i>
            </label>
            <input type="text" id="title" name="title" class="form-control" value="<?php echo htmlspecialchars($lesson['title']); ?>" required>
        </div>
        <div class="mb-3">
            <label for="description" class="form-label">
                Description / déroulement
                <i class="bi bi-question-circle-fill ms-1 text-secondary" role="button" tabindex="0" data-bs-toggle="popover" data-bs-trigger="focus" title="Description" data-bs-content="Précisez ici le déroulement de la séance : mise en route, phases d’apprentissage, dispositifs et consignes."></i>
            </label>
            <textarea id="description" name="description" class="form-control" rows="3"><?php echo htmlspecialchars($lesson['description']); ?></textarea>
        </div>
        <div class="mb-3">
            <label for="fps_select" class="form-label">
                FPS – sélectionner dans la bibliothèque (facultatif)
                <i class="bi bi-question-circle-fill ms-1 text-secondary" role="button" tabindex="0" data-bs-toggle="popover" data-bs-trigger="focus" title="FPS" data-bs-content="La Forme de Pratique Scolaire est la situation de référence ou le jeu servant de support aux apprentissages. Vous pouvez choisir une FPS dans la bibliothèque ou en décrire une vous-même."></i>
            </label>
            <select id="fps_select" name="fps_select" class="form-select">
                <option value="">-- Choisir une forme prédéfinie --</option>
                <?php foreach ($fpsList as $fpsEntry): ?>
                    <option value="<?php echo $fpsEntry['id']; ?>" data-description="<?php echo htmlspecialchars(str_replace('\n', ' ', $fpsEntry['description'])); ?>">
                        <?php echo htmlspecialchars($fpsEntry['ppsad'] . ' – ' . $fpsEntry['name']); ?>
                    </option>
                <?php endforeach; ?>
            </select>
            <div class="form-text">Si vous sélectionnez une FPS, sa description sera copiée ci-dessous et vous pourrez ensuite la modifier.</div>
        </div>
        <div class="mb-3">
            <label for="fps" class="form-label">
                FPS – Forme de Pratique Scolaire (jeu ou situation de référence)
                <i class="bi bi-question-circle-fill ms-1 text-secondary" role="button" tabindex="0" data-bs-toggle="popover" data-bs-trigger="focus" title="FPS" data-bs-content="Décrivez ici la situation de référence utilisée pendant la séance : organisation, rôles, matériel. Cette forme doit permettre de travailler l’OE et les CP."></i>
            </label>
            <textarea id="fps" name="fps" class="form-control" rows="2"><?php echo htmlspecialchars($lesson['fps']); ?></textarea>
        </div>
        <div class="mb-3">
            <label for="cp" class="form-label">
                CP – Contenus prioritaires (séparez-les par des sauts de ligne)
                <i class="bi bi-question-circle-fill ms-1 text-secondary" role="button" tabindex="0" data-bs-toggle="popover" data-bs-trigger="focus" title="Contenus prioritaires" data-bs-content="Les contenus prioritaires regroupent les actions motrices, connaissances et attitudes que les élèves doivent acquérir au cours de cette séance. Listez-les séparés par des retours à la ligne."></i>
            </label>
            <textarea id="cp" name="cp" class="form-control" rows="3"><?php echo htmlspecialchars($lesson['cp']); ?></textarea>
        </div>
        <button type="submit" class="btn btn-primary">Mettre à jour la leçon</button>
    </form>
</div>

<script>
// Lors de la sélection d'une FPS, copier sa description dans le champ dédié
document.getElementById('fps_select').addEventListener('change', function () {
    var select = this;
    var selected = select.options[select.selectedIndex];
    var desc = selected.getAttribute('data-description') || '';
    var txt = desc.replace(/\\n/g, "\n");
    if (desc) {
        document.getElementById('fps').value = txt;
    }
});
</script>

<?php
include 'footer.php';
?>