-- MySQL database schema for the EPS application
--
-- This schema defines the core tables required to run the
-- application described in the specification.  To install the
-- database, import this file into your MySQL server (for example
-- using phpMyAdmin or the mysql CLI).  Make sure to create a
-- database first and use it before running these statements.

-- Table for user accounts.  Teachers log in through this table.
-- You can extend this table with additional fields as needed.
CREATE TABLE IF NOT EXISTS users (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(255) NOT NULL,
  email VARCHAR(255) NOT NULL UNIQUE,
  password VARCHAR(255) NOT NULL,
  role ENUM('teacher','student','admin') NOT NULL DEFAULT 'teacher',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Table for classes.  Each class belongs to a teacher.
CREATE TABLE IF NOT EXISTS classes (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(255) NOT NULL,
  level VARCHAR(255) DEFAULT NULL,
  teacher_id INT NOT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (teacher_id) REFERENCES users(id) ON DELETE CASCADE
);

-- Table for students.  Each student is associated with a class.
CREATE TABLE IF NOT EXISTS students (
  id INT AUTO_INCREMENT PRIMARY KEY,
  class_id INT NOT NULL,
  name VARCHAR(255) NOT NULL,
  -- champs complémentaires pour importer depuis un CSV avec plus de détails
  date_of_birth DATE DEFAULT NULL,
  gender VARCHAR(10) DEFAULT NULL,
  comments TEXT,
  dispense TINYINT(1) DEFAULT 0,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (class_id) REFERENCES classes(id) ON DELETE CASCADE
);

-- Table for cycles (also called sequences).  Each cycle belongs
-- to a class and defines the activity and objective for the cycle.
CREATE TABLE IF NOT EXISTS cycles (
  id INT AUTO_INCREMENT PRIMARY KEY,
  class_id INT NOT NULL,
  name VARCHAR(255) NOT NULL,
  ppsad VARCHAR(255) NOT NULL,
  objective TEXT NOT NULL,
  ecc_type VARCHAR(255) DEFAULT NULL,
  start_date DATE DEFAULT NULL,
  end_date DATE DEFAULT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (class_id) REFERENCES classes(id) ON DELETE CASCADE
);

-- Table for lessons.  Each lesson is part of a cycle and may
-- contain a description, a form of practice (FPS), and contents
-- prioritised (CP).
CREATE TABLE IF NOT EXISTS lessons (
  id INT AUTO_INCREMENT PRIMARY KEY,
  cycle_id INT NOT NULL,
  lesson_date DATE NOT NULL,
  title VARCHAR(255) NOT NULL,
  fps TEXT,
  cp TEXT,
  description TEXT,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (cycle_id) REFERENCES cycles(id) ON DELETE CASCADE
);

-- Table for evaluations.  Stores the score or remarks for each
-- student on a specific lesson.  You can adapt the fields to
-- capture more precise metrics (e.g. multiple criteria, numeric
-- scores) if needed.
CREATE TABLE IF NOT EXISTS evaluations (
  id INT AUTO_INCREMENT PRIMARY KEY,
  lesson_id INT NOT NULL,
  student_id INT NOT NULL,
  score VARCHAR(255) DEFAULT NULL,
  remarks TEXT,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (lesson_id) REFERENCES lessons(id) ON DELETE CASCADE,
  FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE
);

--
-- Extended features
--

-- Table for roles (rôles sociaux).  This table lists the possible
-- roles que les élèves peuvent occuper pendant une leçon (observateur,
-- juge, arbitre, chronométreur, aide, etc.).  Les données sont
-- insérées après la création.
CREATE TABLE IF NOT EXISTS roles (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(255) NOT NULL
);

-- Insérer quelques rôles de base si la table est vide
INSERT INTO roles (name) SELECT * FROM (SELECT 'Observateur') AS tmp
WHERE NOT EXISTS (SELECT 1 FROM roles);
INSERT INTO roles (name) SELECT * FROM (SELECT 'Juge') AS tmp
WHERE NOT EXISTS (SELECT 1 FROM roles WHERE name='Juge');
INSERT INTO roles (name) SELECT * FROM (SELECT 'Chronométreur') AS tmp
WHERE NOT EXISTS (SELECT 1 FROM roles WHERE name='Chronométreur');
INSERT INTO roles (name) SELECT * FROM (SELECT 'Arbitre') AS tmp
WHERE NOT EXISTS (SELECT 1 FROM roles WHERE name='Arbitre');
INSERT INTO roles (name) SELECT * FROM (SELECT 'Aide') AS tmp
WHERE NOT EXISTS (SELECT 1 FROM roles WHERE name='Aide');
INSERT INTO roles (name) SELECT * FROM (SELECT 'Joueur') AS tmp
WHERE NOT EXISTS (SELECT 1 FROM roles WHERE name='Joueur');

-- Table to assign roles to students during a lesson
CREATE TABLE IF NOT EXISTS lesson_roles (
  id INT AUTO_INCREMENT PRIMARY KEY,
  lesson_id INT NOT NULL,
  student_id INT NOT NULL,
  role_id INT NOT NULL,
  FOREIGN KEY (lesson_id) REFERENCES lessons(id) ON DELETE CASCADE,
  FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
  FOREIGN KEY (role_id) REFERENCES roles(id) ON DELETE CASCADE
);

-- Table to store the library of Objects d’Enseignement (OE) pour chaque
-- PPSAD.  Chaque enregistrement fournit un objectif d’enseignement
-- exemplaire ainsi que jusqu’à trois contenus prioritaires associés.
CREATE TABLE IF NOT EXISTS oe_library (
  id INT AUTO_INCREMENT PRIMARY KEY,
  ppsad VARCHAR(255) NOT NULL,
  name VARCHAR(255) NOT NULL,
  description TEXT,
  ecc_type VARCHAR(255),
  cp1 VARCHAR(255),
  cp2 VARCHAR(255),
  cp3 VARCHAR(255)
);

-- Ajouter une colonne oe_id à la table cycles pour lier un cycle à un
-- objet d’enseignement de la bibliothèque.  Cette colonne est
-- facultative afin de ne pas obliger à utiliser la bibliothèque.
ALTER TABLE cycles ADD COLUMN oe_id INT DEFAULT NULL;
ALTER TABLE cycles ADD CONSTRAINT fk_cycles_oe FOREIGN KEY (oe_id) REFERENCES oe_library(id) ON DELETE SET NULL;

-- Ajouter des champs de notation multidimensionnelle à la table
-- evaluations.  Ces colonnes permettent de saisir des notes pour
-- chaque dimension (motricité, intelligibilité, sensibilité,
-- sociabilité) en plus d’un commentaire général.
ALTER TABLE evaluations
  ADD COLUMN motricite INT DEFAULT NULL,
  ADD COLUMN intelligibilite INT DEFAULT NULL,
  ADD COLUMN sensibilite INT DEFAULT NULL,
  ADD COLUMN sociabilite INT DEFAULT NULL,
  ADD COLUMN general_comment TEXT;

-- ------------------------------------------------------------------------------
-- Bibliothèque des Formes de Pratique Scolaire (FPS)
--
-- Afin de soutenir la création de leçons alignées sur les principes du CEDREPS,
-- une bibliothèque optionnelle de FPS est proposée. Chaque entrée décrit une
-- forme de pratique scolaire type pour une activité donnée (PPSAD). Les
-- enseignants peuvent sélectionner une forme existante lors de la création d'une
-- leçon ou saisir librement leur propre description. Les champs
-- supplémentaires permettent de renseigner des contraintes emblématiques et le
-- type d’expérience culturelle (ECC) associé.
CREATE TABLE IF NOT EXISTS fps_library (
  id INT AUTO_INCREMENT PRIMARY KEY,
  ppsad VARCHAR(255) NOT NULL,
  name VARCHAR(255) NOT NULL,
  description TEXT,
  constraints TEXT,
  ecc_type VARCHAR(255),
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);