<?php
require_once __DIR__ . '/functions.php';

// Ensure user logged in
if (!isLoggedIn()) {
    redirect('login.php');
}
$user = currentUser($pdo);

$lessonId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$lesson = getLessonById($pdo, $lessonId);
if (!$lesson) {
    redirect('dashboard.php');
}
$cycle = getCycleById($pdo, $lesson['cycle_id']);
$class = getClassById($pdo, $cycle['class_id']);
// Only authorized roles (teacher, coordinator, inspector, principal) can access the lesson
if (!$class || !userCanAccessClass($pdo, $cycle['class_id'], $user['id'])) {
    redirect('dashboard.php');
}

$students = getStudents($pdo, $class['id']);

// Fetch existing evaluations
$existingEvaluations = [];
$evals = getEvaluations($pdo, $lessonId);
foreach ($evals as $ev) {
    $existingEvaluations[$ev['student_id']] = $ev;
}

// Compute average scores for each dimension (motricite, intelligibilite, sensibilite, sociabilite)
$sumMot = 0; $sumInt = 0; $sumSens = 0; $sumSoc = 0; $count = 0;
foreach ($evals as $ev) {
    // Only count numeric values (not null)
    if (is_numeric($ev['motricite'])) { $sumMot += $ev['motricite']; }
    if (is_numeric($ev['intelligibilite'])) { $sumInt += $ev['intelligibilite']; }
    if (is_numeric($ev['sensibilite'])) { $sumSens += $ev['sensibilite']; }
    if (is_numeric($ev['sociabilite'])) { $sumSoc += $ev['sociabilite']; }
    $count++;
}
if ($count > 0) {
    $avgMot = $sumMot / $count;
    $avgInt = $sumInt / $count;
    $avgSens = $sumSens / $count;
    $avgSoc = $sumSoc / $count;
} else {
    $avgMot = $avgInt = $avgSens = $avgSoc = 0;
}

// Fetch roles and existing assignments
$rolesList = getRoles($pdo);
$lessonRoles = getLessonRoles($pdo, $lessonId);

$message = '';

// Handle evaluation submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['evaluate'])) {
        // Each student line has fields: score[student_id], remarks[student_id]
        // Extract multidimensional scores and comments
        $motricites   = $_POST['motricite']   ?? [];
        $intellis     = $_POST['intelligibilite'] ?? [];
        $sensibilites = $_POST['sensibilite'] ?? [];
        $sociabilites = $_POST['sociabilite'] ?? [];
        $generalComms = $_POST['general_comment'] ?? [];
        foreach ($students as $stu) {
            $sid = $stu['id'];
            $mot = trim($motricites[$sid] ?? '');
            $intell = trim($intellis[$sid] ?? '');
            $sens = trim($sensibilites[$sid] ?? '');
            $soc = trim($sociabilites[$sid] ?? '');
            $comm = trim($generalComms[$sid] ?? '');
            // Cast numeric strings to integers when provided
            $motVal = $mot === '' ? null : (int)$mot;
            $intellVal = $intell === '' ? null : (int)$intell;
            $sensVal = $sens === '' ? null : (int)$sens;
            $socVal = $soc === '' ? null : (int)$soc;
            if (isset($existingEvaluations[$sid])) {
                $stmt = $pdo->prepare('UPDATE evaluations SET motricite=?, intelligibilite=?, sensibilite=?, sociabilite=?, general_comment=? WHERE id = ?');
                $stmt->execute([$motVal, $intellVal, $sensVal, $socVal, $comm, $existingEvaluations[$sid]['id']]);
            } else {
                // Insert if at least one field is not empty
                if ($mot !== '' || $intell !== '' || $sens !== '' || $soc !== '' || $comm !== '') {
                    $stmt = $pdo->prepare('INSERT INTO evaluations (lesson_id, student_id, motricite, intelligibilite, sensibilite, sociabilite, general_comment) VALUES (?, ?, ?, ?, ?, ?, ?)');
                    $stmt->execute([$lessonId, $sid, $motVal, $intellVal, $sensVal, $socVal, $comm]);
                }
            }
        }
        $message = "Évaluations enregistrées.";
        // Refresh existing evaluations
        $evals = getEvaluations($pdo, $lessonId);
        $existingEvaluations = [];
        foreach ($evals as $ev) {
            $existingEvaluations[$ev['student_id']] = $ev;
        }
    }
    // Handle role assignment
    if (isset($_POST['assign_roles'])) {
        $assignments = $_POST['role'] ?? [];
        // Convert values to ints and ensure valid role ids
        $cleanAssign = [];
        foreach ($assignments as $sid => $rid) {
            $rid = (int)$rid;
            if ($rid > 0) {
                $cleanAssign[(int)$sid] = $rid;
            }
        }
        setLessonRoles($pdo, $lessonId, $cleanAssign);
        $lessonRoles = getLessonRoles($pdo, $lessonId);
        $message = ($message ? $message . ' ' : '') . "Rôles enregistrés.";
    }
}

$pageTitle = 'Leçon: ' . $lesson['title'];
include 'header.php';
?>

<div class="mt-4">
    <h2>Détails de la leçon</h2>

    <?php
    // Afficher un lien d'édition pour les enseignants, coordinateurs ou administrateurs
    $roles = getUserRoles($pdo, $_SESSION['user_id']);
    if (in_array('teacher', $roles) || in_array('coordinator', $roles) || in_array('admin', $roles)) {
        echo '<p><a href="edit_lesson.php?id=' . $lessonId . '" class="btn btn-sm btn-warning">Éditer la leçon</a></p>';
    }
    ?>

    <p><strong>Cycle :</strong> <?php echo htmlspecialchars($cycle['name']); ?> (Classe : <?php echo htmlspecialchars($class['name']); ?>)</p>
    <p><strong>Date :</strong> <?php echo htmlspecialchars($lesson['lesson_date']); ?></p>
    <p><strong>Titre :</strong> <?php echo htmlspecialchars($lesson['title']); ?></p>
    <?php
    // Autoriser certaines balises HTML pour les champs riches (les enseignants peuvent insérer du contenu formaté)
    $allowed = '<p><br><ul><ol><li><strong><em><b><i><u><a><img>';
    ?>
    <p><strong>Description :</strong><br>
        <div class="lesson-description"><?php echo strip_tags($lesson['description'], $allowed); ?></div>
    </p>
    <p><strong>FPS :</strong><br>
        <div class="lesson-description"><?php echo strip_tags($lesson['fps'], $allowed); ?></div>
    </p>
    <p><strong>CP :</strong><br>
        <div class="lesson-description"><?php echo strip_tags($lesson['cp'], $allowed); ?></div>
    </p>

    <?php if (!empty($lesson['image'])): ?>
        <div class="my-3">
            <img src="<?php echo htmlspecialchars($lesson['image']); ?>" alt="Illustration de la leçon" class="img-fluid rounded shadow-sm">
        </div>
    <?php endif; ?>

    <!-- Average evaluation chart -->
    <div class="my-4">
        <h3>Statistiques des évaluations</h3>
        <canvas id="evaluationChart" height="150"></canvas>
    </div>

    <h3>Évaluations</h3>
    <?php if ($message): ?>
        <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>
    <form method="post" action="">
        <input type="hidden" name="evaluate" value="1">
        <div class="table-responsive">
            <table class="table table-sm table-striped align-middle datatable">
                <thead class="table-light">
                    <tr>
                        <th>Élève</th>
                        <th>Motricité</th>
                        <th>Intelligibilité</th>
                        <th>Sensibilité</th>
                        <th>Sociabilité</th>
                        <th>Commentaire général</th>
                    </tr>
                </thead>
                <tbody>
                <?php foreach ($students as $stu): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($stu['name']); ?></td>
                        <td><input type="number" name="motricite[<?php echo $stu['id']; ?>]" min="0" max="20" value="<?php echo htmlspecialchars($existingEvaluations[$stu['id']]['motricite'] ?? ''); ?>" class="form-control form-control-sm"></td>
                        <td><input type="number" name="intelligibilite[<?php echo $stu['id']; ?>]" min="0" max="20" value="<?php echo htmlspecialchars($existingEvaluations[$stu['id']]['intelligibilite'] ?? ''); ?>" class="form-control form-control-sm"></td>
                        <td><input type="number" name="sensibilite[<?php echo $stu['id']; ?>]" min="0" max="20" value="<?php echo htmlspecialchars($existingEvaluations[$stu['id']]['sensibilite'] ?? ''); ?>" class="form-control form-control-sm"></td>
                        <td><input type="number" name="sociabilite[<?php echo $stu['id']; ?>]" min="0" max="20" value="<?php echo htmlspecialchars($existingEvaluations[$stu['id']]['sociabilite'] ?? ''); ?>" class="form-control form-control-sm"></td>
                        <td><input type="text" name="general_comment[<?php echo $stu['id']; ?>]" value="<?php echo htmlspecialchars($existingEvaluations[$stu['id']]['general_comment'] ?? ''); ?>" class="form-control form-control-sm"></td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <button type="submit" class="btn btn-primary mt-2">Enregistrer les évaluations</button>
    </form>

    <h3 class="mt-4">Attribution des rôles sociaux</h3>
    <form method="post" action="">
        <input type="hidden" name="assign_roles" value="1">
        <div class="table-responsive">
            <table class="table table-sm table-striped align-middle datatable">
                <thead class="table-light">
                    <tr>
                        <th>Élève</th>
                        <th>Rôle attribué</th>
                    </tr>
                </thead>
                <tbody>
                <?php foreach ($students as $stu): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($stu['name']); ?></td>
                        <td>
                            <select name="role[<?php echo $stu['id']; ?>]" class="form-select form-select-sm">
                                <option value="">-- Aucune attribution --</option>
                                <?php foreach ($rolesList as $role): ?>
                                    <option value="<?php echo $role['id']; ?>" <?php echo (isset($lessonRoles[$stu['id']]) && $lessonRoles[$stu['id']]['role_id'] == $role['id']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($role['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <button type="submit" class="btn btn-primary mt-2">Enregistrer les rôles</button>
        </form>
        <!-- Chart.js library and chart initialization -->
        <script src="https://cdn.jsdelivr.net/npm/chart.js@4.3.0/dist/chart.umd.min.js"></script>
        <script>
        (function() {
            const ctx = document.getElementById('evaluationChart').getContext('2d');
            const data = {
                labels: ['Motricité', 'Intelligibilité', 'Sensibilité', 'Sociabilité'],
                datasets: [{
                    label: 'Moyenne',
                    data: [<?php echo round($avgMot, 2); ?>, <?php echo round($avgInt, 2); ?>, <?php echo round($avgSens, 2); ?>, <?php echo round($avgSoc, 2); ?>],
                    fill: true,
                    tension: 0.3,
                    backgroundColor: 'rgba(54, 162, 235, 0.2)',
                    borderColor: 'rgba(54, 162, 235, 1)'
                }]
            };
            const options = {
                scales: {
                    r: {
                        min: 0,
                        max: 20,
                        ticks: {
                            stepSize: 5
                        }
                    }
                },
                plugins: {
                    legend: { display: false }
                }
            };
            new Chart(ctx, { type: 'radar', data: data, options: options });
        })();
        </script>
    </div>

<?php
include 'footer.php';
?>