<?php
/**
 * ╔══════════════════════════════════════════════════════════════╗
 * ║          CEDREPS ULTIMATE - Configuration Système            ║
 * ║                  Version 2.0 - Révolution EPS                 ║
 * ╚══════════════════════════════════════════════════════════════╝
 * 
 * Application révolutionnaire pour l'enseignement de l'EPS
 * selon la méthodologie CEDREPS - 100% guidée et intuitive
 */

// ===================================================================
// CONFIGURATION DE BASE
// ===================================================================

// Variables d'environnement (à créer dans un fichier .env)
$config = [
    'app' => [
        'name' => 'CEDREPS Ultimate',
        'version' => '2.0',
        'tagline' => 'L\'EPS guidée et intelligente',
        'author' => 'CEDREPS Development Team',
        'year' => 2025
    ],
    
    'database' => [
        'host' => $_ENV['DB_HOST'] ?? 'localhost',
        'name' => $_ENV['DB_NAME'] ?? 'epsweb_appli',
        'user' => $_ENV['DB_USER'] ?? 'epsweb_appli',
        'pass' => $_ENV['DB_PASS'] ?? '',
        'charset' => 'utf8mb4'
    ],
    
    'security' => [
        'session_name' => 'cedreps_ultimate_session',
        'csrf_token_name' => 'cedreps_token',
        'password_min_length' => 8,
        'max_login_attempts' => 5,
        'lockout_duration' => 900, // 15 minutes
        'remember_me_duration' => 2592000 // 30 jours
    ],
    
    'roles' => [
        'SUPER_ADMIN' => [
            'level' => 100,
            'name' => 'Super Administrateur',
            'icon' => 'shield-crown',
            'color' => '#9333EA',
            'description' => 'Accès total au système'
        ],
        'INSPECTEUR' => [
            'level' => 80,
            'name' => 'Inspecteur Académique',
            'icon' => 'user-graduate',
            'color' => '#DC2626',
            'description' => 'Supervision pédagogique académique'
        ],
        'CHEF_ETABLISSEMENT' => [
            'level' => 70,
            'name' => 'Chef d\'Établissement',
            'icon' => 'building-columns',
            'color' => '#EA580C',
            'description' => 'Direction d\'établissement'
        ],
        'COORDINATEUR' => [
            'level' => 60,
            'name' => 'Coordinateur EPS',
            'icon' => 'users-cog',
            'color' => '#0891B2',
            'description' => 'Coordination équipe EPS'
        ],
        'FORMATEUR' => [
            'level' => 50,
            'name' => 'Formateur CEDREPS',
            'icon' => 'chalkboard-teacher',
            'color' => '#059669',
            'description' => 'Formation et accompagnement'
        ],
        'ENSEIGNANT' => [
            'level' => 30,
            'name' => 'Enseignant EPS',
            'icon' => 'user-tie',
            'color' => '#3B82F6',
            'description' => 'Enseignant d\'EPS'
        ],
        'STAGIAIRE' => [
            'level' => 20,
            'name' => 'Stagiaire',
            'icon' => 'user-graduate',
            'color' => '#8B5CF6',
            'description' => 'Enseignant stagiaire'
        ],
        'OBSERVATEUR' => [
            'level' => 10,
            'name' => 'Observateur',
            'icon' => 'eye',
            'color' => '#6B7280',
            'description' => 'Accès en lecture seule'
        ]
    ],
    
    'features' => [
        'guided_mode' => true,          // Mode guidé pour débutants CEDREPS
        'ai_suggestions' => true,        // Suggestions IA pour FPS et OE
        'collaborative_mode' => true,    // Mode collaboratif entre enseignants
        'sharing_system' => true,        // Système de partage avancé
        'analytics' => true,             // Analytics et tableaux de bord
        'export_formats' => ['pdf', 'excel', 'json', 'cedreps'],
        'video_tutorials' => true,       // Tutoriels vidéo intégrés
        'offline_mode' => true,          // Mode hors ligne PWA
        'notifications' => true,         // Système de notifications
        'gamification' => true           // Gamification pour l'apprentissage CEDREPS
    ],
    
    'ui' => [
        'theme' => 'modern',
        'primary_color' => '#3B82F6',
        'secondary_color' => '#8B5CF6',
        'success_color' => '#10B981',
        'warning_color' => '#F59E0B',
        'danger_color' => '#EF4444',
        'info_color' => '#06B6D4',
        'dark_mode' => true,
        'animations' => true,
        'sound_effects' => true
    ],
    
    'cedreps' => [
        'ecc_types' => [
            'ECC1' => [
                'code' => 'ECC1',
                'name' => 'Performance & Dépassement',
                'icon' => 'trophy',
                'color' => '#FF6B6B',
                'gradient' => 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
                'description' => 'Explorer ses limites, se dépasser, performer',
                'mobiles' => 'Dépassement de soi, record personnel, conquête de l\'espace-temps',
                'ppsad_examples' => ['Demi-fond', 'Natation vitesse', 'Relais-vitesse', 'Pentabond']
            ],
            'ECC2' => [
                'code' => 'ECC2',
                'name' => 'Aventure & Exploration',
                'icon' => 'compass',
                'color' => '#4ECDC4',
                'gradient' => 'linear-gradient(135deg, #f093fb 0%, #f5576c 100%)',
                'description' => 'S\'aventurer, explorer l\'inconnu, être autonome',
                'mobiles' => 'Découverte, adaptation au milieu, gestion du risque',
                'ppsad_examples' => ['Escalade', 'Course d\'orientation', 'Sauvetage', 'VTT']
            ],
            'ECC3' => [
                'code' => 'ECC3',
                'name' => 'Arts du Mouvement',
                'icon' => 'theater-masks',
                'color' => '#95E77E',
                'gradient' => 'linear-gradient(135deg, #fa709a 0%, #fee140 100%)',
                'description' => 'S\'exprimer, créer, communiquer par le corps',
                'mobiles' => 'Expression artistique, création, émotion esthétique',
                'ppsad_examples' => ['Danse', 'Arts du cirque', 'Gymnastique rythmique']
            ],
            'ECC4' => [
                'code' => 'ECC4',
                'name' => 'Confrontation & Coopération',
                'icon' => 'handshake',
                'color' => '#FFE66D',
                'gradient' => 'linear-gradient(135deg, #a8edea 0%, #fed6e3 100%)',
                'description' => 'S\'opposer, coopérer, élaborer des stratégies',
                'mobiles' => 'Affrontement codifié, défi mutuel, construction collective',
                'ppsad_examples' => ['Sports collectifs', 'Sports de raquette', 'Sports de combat']
            ],
            'ECC5' => [
                'code' => 'ECC5',
                'name' => 'Découverte & Soin de Soi',
                'icon' => 'heart',
                'color' => '#A8DADC',
                'gradient' => 'linear-gradient(135deg, #ffecd2 0%, #fcb69f 100%)',
                'description' => 'Se connaître, se transformer, prendre soin de soi',
                'mobiles' => 'Bien-être, transformation de soi, harmonie corps-esprit',
                'ppsad_examples' => ['Course en durée', 'Musculation', 'Step', 'Yoga']
            ],
            'ECC6' => [
                'code' => 'ECC6',
                'name' => 'Acrobatie & Défis',
                'icon' => 'star',
                'color' => '#F1FAEE',
                'gradient' => 'linear-gradient(135deg, #a1c4fd 0%, #c2e9fb 100%)',
                'description' => 'Défier la gravité, maîtriser le vertige',
                'mobiles' => 'Jeu avec l\'équilibre, défis gravitaires, sensations fortes',
                'ppsad_examples' => ['Gymnastique', 'Acrosport', 'Trampoline', 'Parkour']
            ]
        ],
        
        'evaluation_dimensions' => [
            'motricite' => [
                'name' => 'Motricité',
                'icon' => 'running',
                'color' => '#EF4444',
                'description' => 'Le corps vital, les transformations motrices, le "pas en avant"'
            ],
            'intelligibilite' => [
                'name' => 'Intelligibilité',
                'icon' => 'brain',
                'color' => '#3B82F6',
                'description' => 'Compréhension, analyse, construction de stratégies'
            ],
            'sensibilite' => [
                'name' => 'Sensibilité',
                'icon' => 'heart',
                'color' => '#F59E0B',
                'description' => 'Ressentis, émotions, rapport sensible au corps'
            ],
            'sociabilite' => [
                'name' => 'Sociabilité',
                'icon' => 'users',
                'color' => '#10B981',
                'description' => 'Rapport aux autres, coopération, communication'
            ]
        ],
        
        'guided_steps' => [
            'DISCOVER' => 'Découverte des concepts',
            'ANALYZE' => 'Analyse du contexte',
            'CHOOSE' => 'Choix de la PPSAD et ECC',
            'TARGET' => 'Ciblage de l\'OE',
            'DESIGN' => 'Conception de la FPS',
            'PLAN' => 'Planification de la séquence',
            'IMPLEMENT' => 'Mise en œuvre',
            'EVALUATE' => 'Évaluation',
            'SHARE' => 'Partage et collaboration'
        ]
    ],
    
    'paths' => [
        'root' => dirname(__DIR__),
        'uploads' => dirname(__DIR__) . '/uploads',
        'temp' => dirname(__DIR__) . '/temp',
        'logs' => dirname(__DIR__) . '/logs',
        'cache' => dirname(__DIR__) . '/cache',
        'exports' => dirname(__DIR__) . '/exports',
        'backups' => dirname(__DIR__) . '/backups'
    ]
];

// ===================================================================
// CONNEXION BASE DE DONNÉES
// ===================================================================

try {
    $pdo = new PDO(
        sprintf(
            "mysql:host=%s;dbname=%s;charset=%s",
            $config['database']['host'],
            $config['database']['name'],
            $config['database']['charset']
        ),
        $config['database']['user'],
        $config['database']['pass'],
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
            PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci"
        ]
    );
} catch (PDOException $e) {
    error_log('[CEDREPS Ultimate] Database Error: ' . $e->getMessage());
    
    // En mode développement, afficher l'erreur
    if ($_ENV['APP_ENV'] === 'development') {
        die('
            <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); 
                        min-height: 100vh; 
                        display: flex; 
                        align-items: center; 
                        justify-content: center; 
                        font-family: -apple-system, BlinkMacSystemFont, sans-serif;">
                <div style="background: white; 
                            padding: 3rem; 
                            border-radius: 1rem; 
                            box-shadow: 0 25px 50px -12px rgba(0,0,0,0.25); 
                            max-width: 500px;">
                    <h1 style="color: #EF4444; margin: 0 0 1rem 0;">⚠️ Erreur de connexion</h1>
                    <p style="color: #6B7280; line-height: 1.6;">
                        Impossible de se connecter à la base de données.<br>
                        Vérifiez votre configuration dans le fichier .env
                    </p>
                    <details style="margin-top: 2rem;">
                        <summary style="cursor: pointer; color: #8B5CF6;">Détails techniques</summary>
                        <pre style="background: #F3F4F6; padding: 1rem; border-radius: 0.5rem; margin-top: 1rem; overflow-x: auto;">
' . htmlspecialchars($e->getMessage()) . '
                        </pre>
                    </details>
                </div>
            </div>
        ');
    } else {
        // En production, page d'erreur générique
        die('Service temporairement indisponible. Veuillez réessayer plus tard.');
    }
}

// ===================================================================
// FONCTIONS GLOBALES
// ===================================================================

/**
 * Obtenir la configuration
 */
function getConfig($key = null) {
    global $config;
    if ($key === null) return $config;
    
    $keys = explode('.', $key);
    $value = $config;
    
    foreach ($keys as $k) {
        if (!isset($value[$k])) return null;
        $value = $value[$k];
    }
    
    return $value;
}

/**
 * Obtenir la connexion PDO
 */
function getDB() {
    global $pdo;
    return $pdo;
}

/**
 * Logger
 */
function logActivity($type, $message, $data = []) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO activity_logs (type, message, data, user_id, ip_address, created_at)
            VALUES (:type, :message, :data, :user_id, :ip, NOW())
        ");
        
        $stmt->execute([
            ':type' => $type,
            ':message' => $message,
            ':data' => json_encode($data),
            ':user_id' => $_SESSION['user_id'] ?? null,
            ':ip' => $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0'
        ]);
    } catch (Exception $e) {
        error_log('[CEDREPS Ultimate] Log Error: ' . $e->getMessage());
    }
}

// ===================================================================
// AUTOLOADER
// ===================================================================

spl_autoload_register(function ($class) {
    $prefix = 'CEDREPS\\';
    $base_dir = __DIR__ . '/classes/';
    
    $len = strlen($prefix);
    if (strncmp($prefix, $class, $len) !== 0) {
        return;
    }
    
    $relative_class = substr($class, $len);
    $file = $base_dir . str_replace('\\', '/', $relative_class) . '.php';
    
    if (file_exists($file)) {
        require $file;
    }
});

// ===================================================================
// INITIALISATION SESSION
// ===================================================================

if (session_status() === PHP_SESSION_NONE) {
    ini_set('session.cookie_httponly', 1);
    ini_set('session.use_only_cookies', 1);
    ini_set('session.cookie_samesite', 'Strict');
    
    if (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') {
        ini_set('session.cookie_secure', 1);
    }
    
    session_name($config['security']['session_name']);
    session_start();
    
    // Régénération périodique de l'ID de session
    if (!isset($_SESSION['last_regeneration'])) {
        $_SESSION['last_regeneration'] = time();
    } elseif (time() - $_SESSION['last_regeneration'] > 1800) {
        session_regenerate_id(true);
        $_SESSION['last_regeneration'] = time();
    }
}

// ===================================================================
// HEADERS DE SÉCURITÉ
// ===================================================================

header('X-Frame-Options: DENY');
header('X-Content-Type-Options: nosniff');
header('X-XSS-Protection: 1; mode=block');
header('Referrer-Policy: strict-origin-when-cross-origin');
header('Permissions-Policy: geolocation=(), microphone=(), camera=()');

// CSP adaptatif
$csp = "default-src 'self'; ";
$csp .= "script-src 'self' 'unsafe-inline' https://cdn.jsdelivr.net https://unpkg.com; ";
$csp .= "style-src 'self' 'unsafe-inline' https://cdn.jsdelivr.net https://fonts.googleapis.com; ";
$csp .= "font-src 'self' https://fonts.gstatic.com; ";
$csp .= "img-src 'self' data: https:; ";
$csp .= "connect-src 'self'; ";
$csp .= "frame-ancestors 'none';";

header("Content-Security-Policy: $csp");

// Force HTTPS en production
if ($_ENV['APP_ENV'] === 'production' && (!isset($_SERVER['HTTPS']) || $_SERVER['HTTPS'] !== 'on')) {
    header('Location: https://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']);
    exit();
}
?>