<?php
/**
 * ╔══════════════════════════════════════════════════════════════╗
 * ║       CEDREPS ULTIMATE - Fonctions Backend Essentielles      ║
 * ║                    Architecture Complète                      ║
 * ╚══════════════════════════════════════════════════════════════╝
 */

require_once 'config.php';

// ===================================================================
// GESTION DE L'AUTHENTIFICATION ET SÉCURITÉ
// ===================================================================

/**
 * Démarrer une session sécurisée
 */
function secureSessionStart() {
    if (session_status() === PHP_SESSION_NONE) {
        $secure = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on';
        $httponly = true;
        
        ini_set('session.use_only_cookies', 1);
        ini_set('session.cookie_samesite', 'Strict');
        
        if ($secure) {
            ini_set('session.cookie_secure', 1);
        }
        
        session_name(getConfig('security.session_name'));
        session_start();
        
        // Régénération périodique de l'ID de session
        if (!isset($_SESSION['last_regeneration'])) {
            $_SESSION['last_regeneration'] = time();
        } elseif (time() - $_SESSION['last_regeneration'] > 1800) {
            session_regenerate_id(true);
            $_SESSION['last_regeneration'] = time();
        }
    }
}

/**
 * Vérifier si l'utilisateur est authentifié
 */
function isAuthenticated() {
    secureSessionStart();
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

/**
 * Obtenir l'utilisateur actuel
 */
function getCurrentUser() {
    if (!isAuthenticated()) {
        return null;
    }
    
    $pdo = getDB();
    $stmt = $pdo->prepare("
        SELECT u.*, 
               GROUP_CONCAT(r.code) as role_codes,
               GROUP_CONCAT(r.name) as role_names
        FROM users u
        LEFT JOIN user_roles ur ON u.id = ur.user_id
        LEFT JOIN roles r ON ur.role_id = r.id
        WHERE u.id = ? AND u.is_active = 1
        GROUP BY u.id
    ");
    $stmt->execute([$_SESSION['user_id']]);
    
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($user) {
        $user['roles'] = $user['role_codes'] ? explode(',', $user['role_codes']) : [];
        $user['role_names'] = $user['role_names'] ? explode(',', $user['role_names']) : [];
    }
    
    return $user;
}

/**
 * Authentifier un utilisateur
 */
function authenticateUser($email, $password) {
    $pdo = getDB();
    
    // Vérifier les tentatives de connexion
    if (!checkLoginAttempts($email)) {
        return ['success' => false, 'error' => 'Trop de tentatives échouées. Réessayez dans 15 minutes.'];
    }
    
    $stmt = $pdo->prepare("
        SELECT id, password, name, is_active 
        FROM users 
        WHERE email = ?
    ");
    $stmt->execute([$email]);
    $user = $stmt->fetch();
    
    if (!$user || !password_verify($password, $user['password'])) {
        recordLoginAttempt($email, false);
        return ['success' => false, 'error' => 'Email ou mot de passe incorrect'];
    }
    
    if (!$user['is_active']) {
        return ['success' => false, 'error' => 'Compte désactivé'];
    }
    
    // Connexion réussie
    recordLoginAttempt($email, true);
    
    secureSessionStart();
    $_SESSION['user_id'] = $user['id'];
    $_SESSION['user_name'] = $user['name'];
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    
    // Mise à jour dernière connexion
    $stmt = $pdo->prepare("UPDATE users SET last_login_at = NOW() WHERE id = ?");
    $stmt->execute([$user['id']]);
    
    logActivity('AUTH', 'Connexion réussie', ['user_id' => $user['id']]);
    
    return ['success' => true, 'user' => $user];
}

/**
 * Vérifier les tentatives de connexion
 */
function checkLoginAttempts($email) {
    $pdo = getDB();
    $ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    $timeLimit = date('Y-m-d H:i:s', strtotime('-15 minutes'));
    
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as attempts 
        FROM login_attempts 
        WHERE (email = ? OR ip_address = ?) 
        AND attempted_at > ? 
        AND success = 0
    ");
    $stmt->execute([$email, $ip, $timeLimit]);
    
    $result = $stmt->fetch();
    return $result['attempts'] < getConfig('security.max_login_attempts');
}

/**
 * Enregistrer une tentative de connexion
 */
function recordLoginAttempt($email, $success = false) {
    $pdo = getDB();
    $stmt = $pdo->prepare("
        INSERT INTO login_attempts (email, ip_address, success) 
        VALUES (?, ?, ?)
    ");
    $stmt->execute([$email, $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0', $success]);
}

// ===================================================================
// GESTION DES RÔLES ET PERMISSIONS
// ===================================================================

/**
 * Obtenir les rôles d'un utilisateur
 */
function getUserRoles($userId) {
    $pdo = getDB();
    $stmt = $pdo->prepare("
        SELECT r.*, ur.expires_at
        FROM user_roles ur
        JOIN roles r ON ur.role_id = r.id
        WHERE ur.user_id = ?
        AND (ur.expires_at IS NULL OR ur.expires_at > NOW())
        ORDER BY r.level DESC
    ");
    $stmt->execute([$userId]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Vérifier si un utilisateur a un rôle spécifique
 */
function userHasRole($userId, $roleCode) {
    $roles = getUserRoles($userId);
    foreach ($roles as $role) {
        if ($role['code'] === $roleCode) {
            return true;
        }
    }
    return false;
}

/**
 * Vérifier si un utilisateur peut accéder à une classe
 */
function userCanAccessClass($userId, $classId) {
    $pdo = getDB();
    
    // Vérifier si l'utilisateur est le propriétaire ou un enseignant associé
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as access
        FROM classes c
        LEFT JOIN class_teachers ct ON c.id = ct.class_id
        WHERE c.id = ?
        AND (c.teacher_id = ? OR ct.user_id = ?)
    ");
    $stmt->execute([$classId, $userId, $userId]);
    $result = $stmt->fetch();
    
    if ($result['access'] > 0) {
        return true;
    }
    
    // Vérifier les rôles spéciaux (inspecteur, chef d'établissement, etc.)
    $userRoles = getUserRoles($userId);
    foreach ($userRoles as $role) {
        if (in_array($role['code'], ['SUPER_ADMIN', 'INSPECTEUR', 'CHEF_ETABLISSEMENT', 'COORDINATEUR'])) {
            return true;
        }
    }
    
    return false;
}

// ===================================================================
// GESTION DES CLASSES ET CYCLES
// ===================================================================

/**
 * Obtenir les classes d'un enseignant
 */
function getTeacherClasses($userId) {
    $pdo = getDB();
    $stmt = $pdo->prepare("
        SELECT DISTINCT c.*, 
               COUNT(DISTINCT s.id) as student_count,
               COUNT(DISTINCT cy.id) as cycle_count
        FROM classes c
        LEFT JOIN class_teachers ct ON c.id = ct.class_id
        LEFT JOIN students s ON c.id = s.class_id
        LEFT JOIN cycles cy ON c.id = cy.class_id
        WHERE (c.teacher_id = ? OR ct.user_id = ?)
        AND c.is_active = 1
        GROUP BY c.id
        ORDER BY c.name
    ");
    $stmt->execute([$userId, $userId]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Créer un nouveau cycle
 */
function createCycle($data, $userId) {
    $pdo = getDB();
    
    try {
        $pdo->beginTransaction();
        
        // Créer le cycle
        $stmt = $pdo->prepare("
            INSERT INTO cycles (
                class_id, name, ppsad, ecc_type, objective,
                fps_name, fps_description, contraintes_emblematiques,
                fonds_culturel_mobiles, fonds_culturel_principes, fonds_culturel_rapport_autre,
                start_date, end_date, nb_lessons_planned, created_by
            ) VALUES (
                :class_id, :name, :ppsad, :ecc_type, :objective,
                :fps_name, :fps_description, :contraintes_emblematiques,
                :fonds_mobiles, :fonds_principes, :fonds_rapport,
                :start_date, :end_date, :nb_lessons, :created_by
            )
        ");
        
        $stmt->execute([
            ':class_id' => $data['class_id'],
            ':name' => $data['name'],
            ':ppsad' => $data['ppsad'],
            ':ecc_type' => $data['ecc_type'],
            ':objective' => $data['objective'],
            ':fps_name' => $data['fps_name'] ?? null,
            ':fps_description' => $data['fps_description'] ?? null,
            ':contraintes_emblematiques' => $data['contraintes_emblematiques'] ?? null,
            ':fonds_mobiles' => $data['fonds_culturel_mobiles'] ?? null,
            ':fonds_principes' => $data['fonds_culturel_principes'] ?? null,
            ':fonds_rapport' => $data['fonds_culturel_rapport_autre'] ?? null,
            ':start_date' => $data['start_date'],
            ':end_date' => $data['end_date'],
            ':nb_lessons' => $data['nb_lessons_planned'] ?? 10,
            ':created_by' => $userId
        ]);
        
        $cycleId = $pdo->lastInsertId();
        
        // Créer automatiquement les leçons planifiées
        if (!empty($data['auto_create_lessons'])) {
            createLessonsForCycle($cycleId, $data['nb_lessons_planned'] ?? 10, $data['start_date']);
        }
        
        // Lier les objets d'enseignement si fournis
        if (!empty($data['oe_ids'])) {
            foreach ($data['oe_ids'] as $index => $oeId) {
                $stmt = $pdo->prepare("
                    INSERT INTO cycle_oe_mappings (cycle_id, oe_id, is_primary)
                    VALUES (?, ?, ?)
                ");
                $stmt->execute([$cycleId, $oeId, $index === 0]);
            }
        }
        
        $pdo->commit();
        
        // Logger l'activité
        logActivity('CYCLE', 'Cycle créé', [
            'cycle_id' => $cycleId,
            'name' => $data['name'],
            'user_id' => $userId
        ]);
        
        // Envoyer une notification
        createNotification($userId, 'CYCLE_CREATED', 
            'Nouveau cycle créé',
            "Le cycle '{$data['name']}' a été créé avec succès",
            ['cycle_id' => $cycleId]
        );
        
        // Mettre à jour les points d'expérience
        updateUserXP($userId, 100, 'cycle_created');
        
        return ['success' => true, 'cycle_id' => $cycleId];
        
    } catch (Exception $e) {
        $pdo->rollBack();
        logActivity('ERROR', 'Erreur création cycle', [
            'error' => $e->getMessage(),
            'user_id' => $userId
        ]);
        return ['success' => false, 'error' => 'Erreur lors de la création du cycle'];
    }
}

/**
 * Créer les leçons pour un cycle
 */
function createLessonsForCycle($cycleId, $nbLessons, $startDate) {
    $pdo = getDB();
    $currentDate = new DateTime($startDate);
    
    for ($i = 1; $i <= $nbLessons; $i++) {
        // Calculer la date de la leçon (1 par semaine)
        if ($i > 1) {
            $currentDate->add(new DateInterval('P7D'));
        }
        
        $stmt = $pdo->prepare("
            INSERT INTO lessons (
                cycle_id, lesson_number, lesson_date, 
                title, status
            ) VALUES (?, ?, ?, ?, 'planned')
        ");
        
        $stmt->execute([
            $cycleId,
            $i,
            $currentDate->format('Y-m-d'),
            "Leçon $i"
        ]);
    }
}

// ===================================================================
// GESTION DES ÉVALUATIONS
// ===================================================================

/**
 * Enregistrer une évaluation
 */
function saveEvaluation($data, $evaluatorId) {
    $pdo = getDB();
    
    try {
        // Vérifier si une évaluation existe déjà
        $stmt = $pdo->prepare("
            SELECT id FROM evaluations 
            WHERE lesson_id = ? AND student_id = ?
        ");
        $stmt->execute([$data['lesson_id'], $data['student_id']]);
        $existing = $stmt->fetch();
        
        if ($existing) {
            // Mise à jour
            $stmt = $pdo->prepare("
                UPDATE evaluations SET
                    motricite = :motricite,
                    intelligibilite = :intelligibilite,
                    sensibilite = :sensibilite,
                    sociabilite = :sociabilite,
                    global_score = :global_score,
                    progress_indicator = :progress,
                    social_role = :social_role,
                    remarks = :remarks,
                    evaluated_by = :evaluator,
                    evaluation_date = NOW()
                WHERE id = :id
            ");
            
            $stmt->execute([
                ':motricite' => $data['motricite'],
                ':intelligibilite' => $data['intelligibilite'],
                ':sensibilite' => $data['sensibilite'],
                ':sociabilite' => $data['sociabilite'],
                ':global_score' => ($data['motricite'] + $data['intelligibilite'] + $data['sensibilite'] + $data['sociabilite']) / 4,
                ':progress' => $data['progress_indicator'] ?? 'en_progres',
                ':social_role' => $data['social_role'] ?? null,
                ':remarks' => $data['remarks'] ?? null,
                ':evaluator' => $evaluatorId,
                ':id' => $existing['id']
            ]);
            
            $evaluationId = $existing['id'];
            
        } else {
            // Création
            $stmt = $pdo->prepare("
                INSERT INTO evaluations (
                    lesson_id, student_id, motricite, intelligibilite,
                    sensibilite, sociabilite, global_score, progress_indicator,
                    social_role, remarks, evaluated_by, evaluation_date
                ) VALUES (
                    :lesson_id, :student_id, :motricite, :intelligibilite,
                    :sensibilite, :sociabilite, :global_score, :progress,
                    :social_role, :remarks, :evaluator, NOW()
                )
            ");
            
            $stmt->execute([
                ':lesson_id' => $data['lesson_id'],
                ':student_id' => $data['student_id'],
                ':motricite' => $data['motricite'],
                ':intelligibilite' => $data['intelligibilite'],
                ':sensibilite' => $data['sensibilite'],
                ':sociabilite' => $data['sociabilite'],
                ':global_score' => ($data['motricite'] + $data['intelligibilite'] + $data['sensibilite'] + $data['sociabilite']) / 4,
                ':progress' => $data['progress_indicator'] ?? 'en_progres',
                ':social_role' => $data['social_role'] ?? null,
                ':remarks' => $data['remarks'] ?? null,
                ':evaluator' => $evaluatorId
            ]);
            
            $evaluationId = $pdo->lastInsertId();
        }
        
        // Enregistrer les dimensions détaillées si fournies
        if (!empty($data['dimensions'])) {
            foreach ($data['dimensions'] as $dimension => $criteria) {
                foreach ($criteria as $criterion => $score) {
                    $stmt = $pdo->prepare("
                        INSERT INTO evaluation_dimensions 
                        (evaluation_id, dimension, criterion, score)
                        VALUES (?, ?, ?, ?)
                        ON DUPLICATE KEY UPDATE score = VALUES(score)
                    ");
                    $stmt->execute([$evaluationId, $dimension, $criterion, $score]);
                }
            }
        }
        
        return ['success' => true, 'evaluation_id' => $evaluationId];
        
    } catch (Exception $e) {
        logActivity('ERROR', 'Erreur évaluation', [
            'error' => $e->getMessage(),
            'evaluator_id' => $evaluatorId
        ]);
        return ['success' => false, 'error' => 'Erreur lors de l\'enregistrement'];
    }
}

// ===================================================================
// CALCUL DE CONFORMITÉ CEDREPS
// ===================================================================

/**
 * Calculer le score de conformité CEDREPS
 */
function calculateCEDREPSConformity($userId) {
    $pdo = getDB();
    $score = 0;
    $criteria = [];
    
    // Critère 1: Objets d'enseignement ciblés (25 points)
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as total,
               SUM(CASE WHEN objective IS NOT NULL AND objective != '' THEN 1 ELSE 0 END) as with_oe
        FROM cycles c
        JOIN classes cl ON c.class_id = cl.id
        WHERE cl.teacher_id = ?
    ");
    $stmt->execute([$userId]);
    $result = $stmt->fetch();
    
    if ($result['total'] > 0) {
        $oeScore = ($result['with_oe'] / $result['total']) * 25;
        $score += $oeScore;
        $criteria['objets_enseignement'] = round($oeScore / 25 * 100);
    }
    
    // Critère 2: FPS avec contraintes emblématiques (25 points)
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as total,
               SUM(CASE WHEN contraintes_emblematiques IS NOT NULL THEN 1 ELSE 0 END) as with_constraints
        FROM cycles c
        JOIN classes cl ON c.class_id = cl.id
        WHERE cl.teacher_id = ?
    ");
    $stmt->execute([$userId]);
    $result = $stmt->fetch();
    
    if ($result['total'] > 0) {
        $fpsScore = ($result['with_constraints'] / $result['total']) * 25;
        $score += $fpsScore;
        $criteria['fps_constraints'] = round($fpsScore / 25 * 100);
    }
    
    // Critère 3: Évaluation des 4 dimensions (25 points)
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as total,
               SUM(CASE WHEN motricite IS NOT NULL 
                        AND intelligibilite IS NOT NULL 
                        AND sensibilite IS NOT NULL 
                        AND sociabilite IS NOT NULL THEN 1 ELSE 0 END) as complete
        FROM evaluations e
        JOIN lessons l ON e.lesson_id = l.id
        JOIN cycles c ON l.cycle_id = c.id
        JOIN classes cl ON c.class_id = cl.id
        WHERE cl.teacher_id = ?
    ");
    $stmt->execute([$userId]);
    $result = $stmt->fetch();
    
    if ($result['total'] > 0) {
        $evalScore = ($result['complete'] / $result['total']) * 25;
        $score += $evalScore;
        $criteria['evaluations_4d'] = round($evalScore / 25 * 100);
    }
    
    // Critère 4: Couverture des 6 ECC (25 points)
    $stmt = $pdo->prepare("
        SELECT COUNT(DISTINCT ecc_type) as ecc_count
        FROM cycles c
        JOIN classes cl ON c.class_id = cl.id
        WHERE cl.teacher_id = ?
        AND ecc_type IS NOT NULL
    ");
    $stmt->execute([$userId]);
    $result = $stmt->fetch();
    
    $eccScore = ($result['ecc_count'] / 6) * 25;
    $score += $eccScore;
    $criteria['ecc_coverage'] = round($eccScore / 25 * 100);
    
    return [
        'score' => round($score),
        'criteria' => $criteria
    ];
}

// ===================================================================
// SYSTÈME DE PARTAGE
// ===================================================================

/**
 * Partager un cycle
 */
function shareCycle($cycleId, $sharedBy, $recipients, $permissions, $message = null, $expiresIn = null) {
    $pdo = getDB();
    $shares = [];
    
    try {
        $pdo->beginTransaction();
        
        foreach ($recipients as $recipientId) {
            // Créer le partage
            $expiresAt = $expiresIn ? date('Y-m-d H:i:s', strtotime("+$expiresIn days")) : null;
            
            $stmt = $pdo->prepare("
                INSERT INTO cycle_shares (
                    cycle_id, shared_by, shared_with, 
                    message, expires_at
                ) VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $cycleId, $sharedBy, $recipientId, 
                $message, $expiresAt
            ]);
            
            $shareId = $pdo->lastInsertId();
            
            // Ajouter les permissions
            foreach ($permissions as $permission) {
                $stmt = $pdo->prepare("
                    INSERT INTO share_permissions (share_id, permission, granted)
                    VALUES (?, ?, 1)
                ");
                $stmt->execute([$shareId, $permission]);
            }
            
            // Notifier le destinataire
            createNotification($recipientId, 'CYCLE_SHARED',
                'Nouveau cycle partagé',
                "Un cycle vous a été partagé",
                ['cycle_id' => $cycleId, 'share_id' => $shareId]
            );
            
            $shares[] = $shareId;
        }
        
        $pdo->commit();
        return ['success' => true, 'shares' => $shares];
        
    } catch (Exception $e) {
        $pdo->rollBack();
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Obtenir les cycles partagés avec un utilisateur
 */
function getSharedCycles($userId) {
    $pdo = getDB();
    $stmt = $pdo->prepare("
        SELECT cs.*, c.name, c.ppsad, c.ecc_type, c.objective,
               cl.name as class_name, u.name as shared_by,
               GROUP_CONCAT(sp.permission) as permissions
        FROM cycle_shares cs
        JOIN cycles c ON cs.cycle_id = c.id
        JOIN classes cl ON c.class_id = cl.id
        JOIN users u ON cs.shared_by = u.id
        LEFT JOIN share_permissions sp ON cs.id = sp.share_id
        WHERE cs.shared_with = ?
        AND cs.status = 'accepted'
        AND (cs.expires_at IS NULL OR cs.expires_at > NOW())
        GROUP BY cs.id
        ORDER BY cs.created_at DESC
    ");
    $stmt->execute([$userId]);
    
    $shares = $stmt->fetchAll(PDO::FETCH_ASSOC);
    foreach ($shares as &$share) {
        $permissions = explode(',', $share['permissions']);
        $share['can_view'] = in_array('view', $permissions);
        $share['can_comment'] = in_array('comment', $permissions);
        $share['can_download'] = in_array('download', $permissions);
        $share['can_duplicate'] = in_array('duplicate', $permissions);
        $share['can_edit'] = in_array('edit', $permissions);
    }
    
    return $shares;
}

/**
 * Obtenir les demandes de partage en attente
 */
function getPendingShareRequests($userId) {
    $pdo = getDB();
    $stmt = $pdo->prepare("
        SELECT cs.*, c.name as cycle_name, u.name as requester_name,
               cs.created_at as requested_at
        FROM cycle_shares cs
        JOIN cycles c ON cs.cycle_id = c.id
        JOIN users u ON cs.shared_with = u.id
        WHERE cs.shared_by = ?
        AND cs.status = 'pending'
        ORDER BY cs.created_at DESC
    ");
    $stmt->execute([$userId]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Obtenir mes partages
 */
function getMyShares($userId) {
    $pdo = getDB();
    $stmt = $pdo->prepare("
        SELECT cs.*, c.name as cycle_name, u.name as shared_with_name
        FROM cycle_shares cs
        JOIN cycles c ON cs.cycle_id = c.id
        JOIN users u ON cs.shared_with = u.id
        WHERE cs.shared_by = ?
        ORDER BY cs.created_at DESC
    ");
    $stmt->execute([$userId]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// ===================================================================
// SYSTÈME DE NOTIFICATIONS
// ===================================================================

/**
 * Créer une notification
 */
function createNotification($userId, $type, $title, $message, $data = []) {
    $pdo = getDB();
    
    // Déterminer l'icône et la couleur selon le type
    $icons = [
        'CYCLE_CREATED' => 'fa-sync',
        'CYCLE_SHARED' => 'fa-share-alt',
        'EVALUATION_COMPLETED' => 'fa-check-circle',
        'ACHIEVEMENT_UNLOCKED' => 'fa-trophy',
        'COMMENT_RECEIVED' => 'fa-comment'
    ];
    
    $colors = [
        'CYCLE_CREATED' => '#3B82F6',
        'CYCLE_SHARED' => '#8B5CF6',
        'EVALUATION_COMPLETED' => '#10B981',
        'ACHIEVEMENT_UNLOCKED' => '#F59E0B',
        'COMMENT_RECEIVED' => '#06B6D4'
    ];
    
    $stmt = $pdo->prepare("
        INSERT INTO notifications (
            user_id, type, title, message, data,
            icon, color
        ) VALUES (?, ?, ?, ?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $userId,
        $type,
        $title,
        $message,
        json_encode($data),
        $icons[$type] ?? 'fa-bell',
        $colors[$type] ?? '#6B7280'
    ]);
    
    return $pdo->lastInsertId();
}

// ===================================================================
// SYSTÈME DE GAMIFICATION
// ===================================================================

/**
 * Mettre à jour l'expérience utilisateur
 */
function updateUserXP($userId, $xpAmount, $reason) {
    $pdo = getDB();
    
    // Récupérer le niveau actuel
    $stmt = $pdo->prepare("SELECT cedreps_level, cedreps_xp FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    $user = $stmt->fetch();
    
    $newXP = $user['cedreps_xp'] + $xpAmount;
    $newLevel = calculateLevel($newXP);
    
    // Mise à jour
    $stmt = $pdo->prepare("
        UPDATE users 
        SET cedreps_xp = ?, cedreps_level = ?
        WHERE id = ?
    ");
    $stmt->execute([$newXP, $newLevel, $userId]);
    
    // Si niveau supérieur, créer notification
    if ($newLevel > $user['cedreps_level']) {
        createNotification($userId, 'LEVEL_UP',
            'Niveau supérieur atteint !',
            "Félicitations ! Vous avez atteint le niveau $newLevel",
            ['old_level' => $user['cedreps_level'], 'new_level' => $newLevel]
        );
    }
    
    // Logger l'activité
    logActivity('GAMIFICATION', 'XP gagné', [
        'user_id' => $userId,
        'xp_amount' => $xpAmount,
        'reason' => $reason,
        'new_total' => $newXP
    ]);
}

/**
 * Calculer le niveau basé sur l'XP
 */
function calculateLevel($xp) {
    $levels = [
        0 => 0,
        1 => 100,
        2 => 300,
        3 => 600,
        4 => 1000,
        5 => 1500,
        6 => 2100,
        7 => 2800,
        8 => 3600,
        9 => 4500,
        10 => 5500
    ];
    
    $level = 0;
    foreach ($levels as $lvl => $xpRequired) {
        if ($xp >= $xpRequired) {
            $level = $lvl;
        } else {
            break;
        }
    }
    
    return $level;
}

/**
 * Vérifier et débloquer les achievements
 */
function checkAchievements($userId) {
    $pdo = getDB();
    $unlocked = [];
    
    // Vérifier achievement "Premier Cycle"
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as count 
        FROM cycles c
        JOIN classes cl ON c.class_id = cl.id
        WHERE cl.teacher_id = ?
    ");
    $stmt->execute([$userId]);
    $result = $stmt->fetch();
    
    if ($result['count'] >= 1) {
        unlockAchievement($userId, 'FIRST_CYCLE');
        $unlocked[] = 'FIRST_CYCLE';
    }
    
    // Vérifier achievement "5 FPS Créées"
    if ($result['count'] >= 5) {
        unlockAchievement($userId, 'FIVE_FPS');
        $unlocked[] = 'FIVE_FPS';
    }
    
    // Vérifier achievement "Explorateur ECC"
    $stmt = $pdo->prepare("
        SELECT COUNT(DISTINCT ecc_type) as ecc_count
        FROM cycles c
        JOIN classes cl ON c.class_id = cl.id
        WHERE cl.teacher_id = ?
        AND ecc_type IS NOT NULL
    ");
    $stmt->execute([$userId]);
    $result = $stmt->fetch();
    
    if ($result['ecc_count'] >= 6) {
        unlockAchievement($userId, 'ALL_ECC');
        $unlocked[] = 'ALL_ECC';
    }
    
    // Vérifier achievement "Conformité Parfaite"
    $conformity = calculateCEDREPSConformity($userId);
    if ($conformity['score'] >= 100) {
        unlockAchievement($userId, 'PERFECT_CONFORMITY');
        $unlocked[] = 'PERFECT_CONFORMITY';
    }
    
    return $unlocked;
}

/**
 * Débloquer un achievement
 */
function unlockAchievement($userId, $achievementCode) {
    $pdo = getDB();
    
    // Vérifier si déjà débloqué
    $stmt = $pdo->prepare("
        SELECT ua.*, a.name, a.xp_reward
        FROM user_achievements ua
        JOIN achievements a ON ua.achievement_id = a.id
        WHERE ua.user_id = ? AND a.code = ?
    ");
    $stmt->execute([$userId, $achievementCode]);
    
    if (!$stmt->fetch()) {
        // Obtenir l'achievement
        $stmt = $pdo->prepare("SELECT * FROM achievements WHERE code = ?");
        $stmt->execute([$achievementCode]);
        $achievement = $stmt->fetch();
        
        if ($achievement) {
            // Débloquer
            $stmt = $pdo->prepare("
                INSERT INTO user_achievements (user_id, achievement_id)
                VALUES (?, ?)
            ");
            $stmt->execute([$userId, $achievement['id']]);
            
            // Ajouter l'XP
            updateUserXP($userId, $achievement['xp_reward'], 'achievement_' . $achievementCode);
            
            // Notifier
            createNotification($userId, 'ACHIEVEMENT_UNLOCKED',
                'Achievement débloqué !',
                $achievement['name'],
                ['achievement_code' => $achievementCode]
            );
            
            return true;
        }
    }
    
    return false;
}

// ===================================================================
// STATISTIQUES ET ANALYTICS
// ===================================================================

/**
 * Obtenir les statistiques du tableau de bord
 */
function getDashboardStats($userId) {
    $pdo = getDB();
    $stats = [];
    
    // Total cycles
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as total
        FROM cycles c
        JOIN classes cl ON c.class_id = cl.id
        WHERE cl.teacher_id = ?
    ");
    $stmt->execute([$userId]);
    $stats['total_cycles'] = $stmt->fetchColumn();
    
    // Total leçons
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as total
        FROM lessons l
        JOIN cycles c ON l.cycle_id = c.id
        JOIN classes cl ON c.class_id = cl.id
        WHERE cl.teacher_id = ?
    ");
    $stmt->execute([$userId]);
    $stats['total_lessons'] = $stmt->fetchColumn();
    
    // Total élèves
    $stmt = $pdo->prepare("
        SELECT COUNT(DISTINCT s.id) as total
        FROM students s
        JOIN classes cl ON s.class_id = cl.id
        WHERE cl.teacher_id = ?
    ");
    $stmt->execute([$userId]);
    $stats['total_students'] = $stmt->fetchColumn();
    
    // Total FPS
    $stmt = $pdo->prepare("
        SELECT COUNT(DISTINCT fps_name) as total
        FROM cycles c
        JOIN classes cl ON c.class_id = cl.id
        WHERE cl.teacher_id = ?
        AND fps_name IS NOT NULL
    ");
    $stmt->execute([$userId]);
    $stats['total_fps'] = $stmt->fetchColumn();
    
    return $stats;
}

// ===================================================================
// VALIDATION ET SÉCURITÉ
// ===================================================================

/**
 * Générer un token CSRF
 */
function generateCSRFToken() {
    if (empty($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * Vérifier le token CSRF
 */
function verifyCSRFToken($token) {
    if (!isset($_SESSION['csrf_token']) || $token !== $_SESSION['csrf_token']) {
        throw new Exception('Token CSRF invalide');
    }
    return true;
}

/**
 * Nettoyer et valider les entrées
 */
function sanitizeInput($input, $type = 'string') {
    switch ($type) {
        case 'email':
            return filter_var($input, FILTER_SANITIZE_EMAIL);
        case 'int':
            return filter_var($input, FILTER_SANITIZE_NUMBER_INT);
        case 'float':
            return filter_var($input, FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION);
        case 'url':
            return filter_var($input, FILTER_SANITIZE_URL);
        case 'html':
            return htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
        default:
            return htmlspecialchars(strip_tags($input), ENT_QUOTES, 'UTF-8');
    }
}

/**
 * Fonction d'échappement pour l'affichage
 */
function e($string) {
    return htmlspecialchars($string, ENT_QUOTES, 'UTF-8');
}
?>