# Guide d'Amélioration de la Sécurité - Application EPS-CEDREPS

## 🔐 Corrections de Sécurité Prioritaires

### 1. Protection des Identifiants de Base de Données

#### ❌ Code Actuel (DANGEREUX)
```php
// config.php
$db_host = 'localhost';
$db_name = 'epsweb_appli';
$db_user = 'epsweb_appli';
$db_pass = 'MF03444janvierA'; // Mot de passe en clair !
```

#### ✅ Solution Recommandée

**Étape 1: Créer un fichier `.env` (à la racine)**
```env
DB_HOST=localhost
DB_NAME=epsweb_appli
DB_USER=epsweb_appli
DB_PASSWORD=MF03444janvierA
```

**Étape 2: Ajouter `.env` au `.gitignore`**
```gitignore
.env
config.php
```

**Étape 3: Nouveau `config.php`**
```php
<?php
// Charger les variables d'environnement
function loadEnv($path = '.env') {
    if (!file_exists($path)) {
        die('Fichier .env non trouvé. Veuillez créer le fichier de configuration.');
    }
    
    $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        if (strpos(trim($line), '#') === 0) continue;
        
        list($name, $value) = explode('=', $line, 2);
        $name = trim($name);
        $value = trim($value);
        
        if (!array_key_exists($name, $_ENV)) {
            $_ENV[$name] = $value;
        }
    }
}

loadEnv(__DIR__ . '/.env');

$db_host = $_ENV['DB_HOST'] ?? 'localhost';
$db_name = $_ENV['DB_NAME'] ?? '';
$db_user = $_ENV['DB_USER'] ?? '';
$db_pass = $_ENV['DB_PASSWORD'] ?? '';

if (empty($db_name) || empty($db_user) || empty($db_pass)) {
    die('Configuration de base de données incomplète.');
}

try {
    $pdo = new PDO(
        "mysql:host=$db_host;dbname=$db_name;charset=utf8mb4",
        $db_user,
        $db_pass,
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false
        ]
    );
} catch (PDOException $e) {
    error_log('Database connection error: ' . $e->getMessage());
    die('Erreur de connexion à la base de données.');
}
?>
```

---

### 2. Protection CSRF (Cross-Site Request Forgery)

#### ✅ Ajouter dans `functions.php`

```php
/**
 * Générer un token CSRF
 */
function generateCSRFToken() {
    if (empty($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * Vérifier le token CSRF
 */
function verifyCSRFToken($token) {
    if (!isset($_SESSION['csrf_token']) || $token !== $_SESSION['csrf_token']) {
        die('Erreur de sécurité: Token CSRF invalide');
    }
    return true;
}

/**
 * Créer un champ hidden pour le token CSRF
 */
function csrfField() {
    return '<input type="hidden" name="csrf_token" value="' . htmlspecialchars(generateCSRFToken()) . '">';
}
```

#### ✅ Utilisation dans les formulaires

**Exemple: `create_class.php`**
```php
<form method="POST" action="create_class.php">
    <?php echo csrfField(); ?>
    
    <div class="mb-3">
        <label for="name" class="form-label">Nom de la classe</label>
        <input type="text" class="form-control" id="name" name="name" required>
    </div>
    
    <button type="submit" class="btn btn-primary">Créer</button>
</form>

<?php
// Traitement du formulaire
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Vérifier le token CSRF
    verifyCSRFToken($_POST['csrf_token'] ?? '');
    
    // Continuer le traitement...
    $name = filter_input(INPUT_POST, 'name', FILTER_SANITIZE_STRING);
    // ...
}
?>
```

---

### 3. Validation et Sanitisation des Données

#### ✅ Ajouter dans `functions.php`

```php
/**
 * Classe de validation
 */
class Validator {
    private $errors = [];
    private $data = [];
    
    public function __construct($data) {
        $this->data = $data;
    }
    
    public function validate($field, $rules) {
        $value = $this->data[$field] ?? null;
        $rules = explode('|', $rules);
        
        foreach ($rules as $rule) {
            $params = [];
            if (strpos($rule, ':') !== false) {
                list($rule, $paramStr) = explode(':', $rule);
                $params = explode(',', $paramStr);
            }
            
            $method = 'validate' . ucfirst($rule);
            if (method_exists($this, $method)) {
                $this->$method($field, $value, $params);
            }
        }
        
        return $this;
    }
    
    private function validateRequired($field, $value, $params) {
        if (empty($value)) {
            $this->errors[$field][] = "Le champ $field est requis.";
        }
    }
    
    private function validateEmail($field, $value, $params) {
        if (!filter_var($value, FILTER_VALIDATE_EMAIL)) {
            $this->errors[$field][] = "L'email n'est pas valide.";
        }
    }
    
    private function validateMin($field, $value, $params) {
        $min = $params[0] ?? 0;
        if (strlen($value) < $min) {
            $this->errors[$field][] = "Le champ $field doit contenir au moins $min caractères.";
        }
    }
    
    private function validateMax($field, $value, $params) {
        $max = $params[0] ?? 255;
        if (strlen($value) > $max) {
            $this->errors[$field][] = "Le champ $field ne peut pas dépasser $max caractères.";
        }
    }
    
    private function validateNumeric($field, $value, $params) {
        if (!is_numeric($value)) {
            $this->errors[$field][] = "Le champ $field doit être numérique.";
        }
    }
    
    private function validateDate($field, $value, $params) {
        $format = $params[0] ?? 'Y-m-d';
        $d = DateTime::createFromFormat($format, $value);
        if (!$d || $d->format($format) !== $value) {
            $this->errors[$field][] = "La date n'est pas au bon format.";
        }
    }
    
    public function fails() {
        return !empty($this->errors);
    }
    
    public function errors() {
        return $this->errors;
    }
    
    public function getSanitized($field, $filter = FILTER_SANITIZE_STRING) {
        return filter_var($this->data[$field] ?? '', $filter);
    }
}

/**
 * Fonction helper pour échapper les sorties HTML
 */
function e($string) {
    return htmlspecialchars($string, ENT_QUOTES, 'UTF-8');
}
```

#### ✅ Utilisation de la validation

```php
// Exemple dans register.php
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verifyCSRFToken($_POST['csrf_token'] ?? '');
    
    $validator = new Validator($_POST);
    $validator->validate('name', 'required|min:2|max:255')
              ->validate('email', 'required|email')
              ->validate('password', 'required|min:8')
              ->validate('role', 'required');
    
    if ($validator->fails()) {
        $errors = $validator->errors();
        // Afficher les erreurs
        foreach ($errors as $field => $messages) {
            foreach ($messages as $message) {
                echo "<div class='alert alert-danger'>$message</div>";
            }
        }
    } else {
        // Données validées et sécurisées
        $name = $validator->getSanitized('name');
        $email = $validator->getSanitized('email', FILTER_SANITIZE_EMAIL);
        $password = password_hash($_POST['password'], PASSWORD_DEFAULT);
        $role = $validator->getSanitized('role');
        
        // Insertion en base de données...
    }
}
```

---

### 4. Headers de Sécurité

#### ✅ Ajouter dans `header.php` (au tout début)

```php
<?php
// Headers de sécurité
header('X-Frame-Options: SAMEORIGIN');
header('X-Content-Type-Options: nosniff');
header('X-XSS-Protection: 1; mode=block');
header('Referrer-Policy: strict-origin-when-cross-origin');
header('Permissions-Policy: geolocation=(), microphone=(), camera=()');

// Content Security Policy (adapter selon vos besoins)
$csp = "default-src 'self'; ";
$csp .= "script-src 'self' 'unsafe-inline' https://cdn.jsdelivr.net https://cdn.datatables.net https://cdnjs.cloudflare.com; ";
$csp .= "style-src 'self' 'unsafe-inline' https://cdn.jsdelivr.net https://cdn.datatables.net; ";
$csp .= "img-src 'self' data: https:; ";
$csp .= "font-src 'self' https://cdn.jsdelivr.net; ";
$csp .= "connect-src 'self'; ";
$csp .= "frame-ancestors 'self';";

header("Content-Security-Policy: $csp");

// Force HTTPS
if (!isset($_SERVER['HTTPS']) || $_SERVER['HTTPS'] !== 'on') {
    if ($_SERVER['HTTP_HOST'] !== 'localhost') {
        header('Location: https://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']);
        exit();
    }
}
?>
```

---

### 5. Protection contre les Injections SQL (Améliorations)

#### ✅ Requêtes Préparées Sécurisées

```php
/**
 * Fonction sécurisée pour récupérer une classe
 */
function getClassById($pdo, $classId, $userId) {
    // Vérifier que l'ID est numérique
    if (!is_numeric($classId)) {
        return null;
    }
    
    // Requête préparée avec vérification des droits
    $stmt = $pdo->prepare('
        SELECT c.* 
        FROM classes c
        LEFT JOIN class_teachers ct ON c.id = ct.class_id
        WHERE c.id = :class_id 
        AND (c.teacher_id = :user_id OR ct.user_id = :user_id2)
        LIMIT 1
    ');
    
    $stmt->execute([
        ':class_id' => $classId,
        ':user_id' => $userId,
        ':user_id2' => $userId
    ]);
    
    return $stmt->fetch();
}

/**
 * Fonction sécurisée pour les insertions
 */
function createClass($pdo, $data, $userId) {
    // Validation préalable
    $validator = new Validator($data);
    $validator->validate('name', 'required|max:255')
              ->validate('level', 'max:255');
    
    if ($validator->fails()) {
        return ['success' => false, 'errors' => $validator->errors()];
    }
    
    try {
        $pdo->beginTransaction();
        
        // Insertion sécurisée
        $stmt = $pdo->prepare('
            INSERT INTO classes (name, level, teacher_id, created_at) 
            VALUES (:name, :level, :teacher_id, NOW())
        ');
        
        $stmt->execute([
            ':name' => $validator->getSanitized('name'),
            ':level' => $validator->getSanitized('level'),
            ':teacher_id' => $userId
        ]);
        
        $classId = $pdo->lastInsertId();
        
        // Ajouter l'enseignant dans class_teachers
        $stmt = $pdo->prepare('
            INSERT INTO class_teachers (class_id, user_id) 
            VALUES (:class_id, :user_id)
        ');
        
        $stmt->execute([
            ':class_id' => $classId,
            ':user_id' => $userId
        ]);
        
        $pdo->commit();
        
        return ['success' => true, 'id' => $classId];
        
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log('Erreur création classe: ' . $e->getMessage());
        return ['success' => false, 'error' => 'Erreur lors de la création'];
    }
}
```

---

### 6. Gestion Sécurisée des Sessions

#### ✅ Améliorer `functions.php`

```php
/**
 * Démarrer une session sécurisée
 */
function secureSessionStart() {
    $session_name = 'eps_session';
    $secure = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on';
    $httponly = true;
    
    ini_set('session.use_only_cookies', 1);
    
    $cookieParams = session_get_cookie_params();
    session_set_cookie_params(
        $cookieParams["lifetime"],
        $cookieParams["path"],
        $cookieParams["domain"],
        $secure,
        $httponly
    );
    
    session_name($session_name);
    session_start();
    
    // Régénérer l'ID de session périodiquement
    if (!isset($_SESSION['last_regeneration'])) {
        $_SESSION['last_regeneration'] = time();
    } elseif (time() - $_SESSION['last_regeneration'] > 300) {
        session_regenerate_id(true);
        $_SESSION['last_regeneration'] = time();
    }
}

// Remplacer session_start() par secureSessionStart() dans tous les fichiers
```

---

### 7. Limitation des Tentatives de Connexion

#### ✅ Ajouter la protection contre le brute force

```php
/**
 * Table SQL à ajouter
 */
CREATE TABLE login_attempts (
    id INT AUTO_INCREMENT PRIMARY KEY,
    email VARCHAR(255),
    ip_address VARCHAR(45),
    attempted_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_email_ip (email, ip_address, attempted_at)
);

/**
 * Fonction de vérification des tentatives
 */
function checkLoginAttempts($pdo, $email) {
    $ip = $_SERVER['REMOTE_ADDR'];
    $timeLimit = date('Y-m-d H:i:s', strtotime('-15 minutes'));
    
    // Compter les tentatives récentes
    $stmt = $pdo->prepare('
        SELECT COUNT(*) as attempts 
        FROM login_attempts 
        WHERE (email = :email OR ip_address = :ip) 
        AND attempted_at > :time_limit
    ');
    
    $stmt->execute([
        ':email' => $email,
        ':ip' => $ip,
        ':time_limit' => $timeLimit
    ]);
    
    $result = $stmt->fetch();
    
    if ($result['attempts'] >= 5) {
        return false; // Trop de tentatives
    }
    
    return true;
}

/**
 * Enregistrer une tentative de connexion
 */
function recordLoginAttempt($pdo, $email) {
    $stmt = $pdo->prepare('
        INSERT INTO login_attempts (email, ip_address) 
        VALUES (:email, :ip)
    ');
    
    $stmt->execute([
        ':email' => $email,
        ':ip' => $_SERVER['REMOTE_ADDR']
    ]);
}

/**
 * Nettoyer les anciennes tentatives
 */
function cleanOldAttempts($pdo) {
    $stmt = $pdo->prepare('
        DELETE FROM login_attempts 
        WHERE attempted_at < DATE_SUB(NOW(), INTERVAL 1 DAY)
    ');
    $stmt->execute();
}
```

---

### 8. Journalisation Sécurisée

#### ✅ Système de logs

```php
/**
 * Classe de journalisation
 */
class SecurityLogger {
    private $logFile;
    
    public function __construct($logFile = '/logs/security.log') {
        $this->logFile = __DIR__ . $logFile;
        $logDir = dirname($this->logFile);
        
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
    }
    
    public function log($level, $message, $context = []) {
        $timestamp = date('Y-m-d H:i:s');
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'CLI';
        $userId = $_SESSION['user_id'] ?? 'anonymous';
        
        $logEntry = sprintf(
            "[%s] %s | IP: %s | User: %s | %s | Context: %s\n",
            $timestamp,
            strtoupper($level),
            $ip,
            $userId,
            $message,
            json_encode($context)
        );
        
        file_put_contents($this->logFile, $logEntry, FILE_APPEND | LOCK_EX);
        
        // Rotation des logs si le fichier devient trop gros
        if (filesize($this->logFile) > 10485760) { // 10 MB
            $this->rotateLog();
        }
    }
    
    private function rotateLog() {
        $newName = $this->logFile . '.' . date('Y-m-d-H-i-s');
        rename($this->logFile, $newName);
        
        // Garder seulement les 10 derniers fichiers
        $files = glob(dirname($this->logFile) . '/*.log.*');
        if (count($files) > 10) {
            usort($files, function($a, $b) {
                return filemtime($a) - filemtime($b);
            });
            
            $toDelete = array_slice($files, 0, count($files) - 10);
            foreach ($toDelete as $file) {
                unlink($file);
            }
        }
    }
    
    public function logSecurity($message, $context = []) {
        $this->log('security', $message, $context);
    }
    
    public function logError($message, $context = []) {
        $this->log('error', $message, $context);
    }
    
    public function logInfo($message, $context = []) {
        $this->log('info', $message, $context);
    }
}

// Utilisation
$logger = new SecurityLogger();
$logger->logSecurity('Tentative de connexion échouée', ['email' => $email]);
```

---

## 📋 Checklist de Sécurité

### Implémentation Immédiate
- [ ] Migrer les identifiants vers des variables d'environnement
- [ ] Implémenter les tokens CSRF sur tous les formulaires
- [ ] Ajouter la validation côté serveur
- [ ] Configurer les headers de sécurité
- [ ] Implémenter la limitation des tentatives de connexion

### Implémentation à Court Terme
- [ ] Mettre en place le système de logs
- [ ] Ajouter un système de captcha après 3 tentatives échouées
- [ ] Implémenter 2FA (authentification à deux facteurs)
- [ ] Audit de sécurité complet du code
- [ ] Tests de pénétration

### Maintenance Continue
- [ ] Mise à jour régulière des dépendances
- [ ] Révision périodique des logs de sécurité
- [ ] Formation continue sur les bonnes pratiques
- [ ] Backup régulier et testé
- [ ] Plan de réponse aux incidents

---

## 🔒 Fichier `.htaccess` Recommandé

```apache
# Protection du répertoire
Options -Indexes
Options -MultiViews

# Protection des fichiers sensibles
<FilesMatch "^\.env|\.git|composer\.(json|lock)|package\.(json|lock)$">
    Order Allow,Deny
    Deny from all
</FilesMatch>

# Protection contre les injections
RewriteEngine On
RewriteCond %{QUERY_STRING} (\<|%3C).*script.*(\>|%3E) [NC,OR]
RewriteCond %{QUERY_STRING} GLOBALS(=|\[|\%[0-9A-Z]{0,2}) [OR]
RewriteCond %{QUERY_STRING} _REQUEST(=|\[|\%[0-9A-Z]{0,2})
RewriteRule .* - [F,L]

# Force HTTPS
RewriteCond %{HTTPS} !=on
RewriteRule ^(.*)$ https://%{HTTP_HOST}/$1 [R=301,L]

# Headers de sécurité supplémentaires
Header set X-Frame-Options "SAMEORIGIN"
Header set X-Content-Type-Options "nosniff"
Header set X-XSS-Protection "1; mode=block"

# Compression
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/html text/css text/javascript application/javascript
</IfModule>

# Cache
<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType image/jpg "access plus 1 year"
    ExpiresByType image/png "access plus 1 year"
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType application/javascript "access plus 1 month"
</IfModule>
```

---

## 📚 Ressources Utiles

- [OWASP Top 10](https://owasp.org/www-project-top-ten/)
- [PHP Security Guide](https://www.php.net/manual/en/security.php)
- [OWASP PHP Security Cheat Sheet](https://cheatsheetseries.owasp.org/cheatsheets/PHP_Configuration_Cheat_Sheet.html)
- [Content Security Policy](https://developer.mozilla.org/en-US/docs/Web/HTTP/CSP)

---

*Ce guide vous aidera à sécuriser votre application. Implémentez ces mesures progressivement en commençant par les plus critiques.*