<?php
/**
 * ╔══════════════════════════════════════════════════════════════╗
 * ║      DIAGNOSTIC ET CORRECTION DES PERMISSIONS                 ║
 * ║          Résolution de l'erreur 403 Forbidden                 ║
 * ╚══════════════════════════════════════════════════════════════╝
 */

// Ce fichier aide à diagnostiquer et résoudre les problèmes de permissions
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Diagnostic des Permissions - CEDREPS Ultimate</title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 40px 20px;
            margin: 0;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            background: white;
            border-radius: 20px;
            padding: 40px;
            box-shadow: 0 25px 50px -12px rgba(0,0,0,0.25);
        }
        h1 {
            color: #1F2937;
            margin-bottom: 10px;
        }
        .subtitle {
            color: #6B7280;
            margin-bottom: 30px;
        }
        .diagnostic-item {
            background: #F9FAFB;
            padding: 20px;
            border-radius: 12px;
            margin-bottom: 20px;
            border-left: 4px solid #E5E7EB;
        }
        .diagnostic-item.success {
            border-color: #10B981;
            background: #F0FDF4;
        }
        .diagnostic-item.warning {
            border-color: #F59E0B;
            background: #FFFBEB;
        }
        .diagnostic-item.error {
            border-color: #EF4444;
            background: #FEF2F2;
        }
        .status-icon {
            font-size: 20px;
            margin-right: 10px;
        }
        .code-block {
            background: #1F2937;
            color: #10B981;
            padding: 15px;
            border-radius: 8px;
            margin: 15px 0;
            font-family: monospace;
            font-size: 14px;
            overflow-x: auto;
        }
        .btn {
            display: inline-block;
            padding: 12px 24px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-weight: 600;
            margin: 10px 5px;
        }
        .btn:hover {
            transform: translateY(-2px);
        }
        .solution-box {
            background: #EFF6FF;
            border: 1px solid #3B82F6;
            padding: 20px;
            border-radius: 12px;
            margin: 30px 0;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
        }
        th, td {
            padding: 10px;
            text-align: left;
            border-bottom: 1px solid #E5E7EB;
        }
        th {
            background: #F3F4F6;
            font-weight: 600;
        }
        .permission-code {
            font-family: monospace;
            background: #F3F4F6;
            padding: 2px 6px;
            border-radius: 4px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔧 Diagnostic et Correction des Permissions</h1>
        <p class="subtitle">Résolution de l'erreur 403 Forbidden sur /app_v5/</p>
        
        <?php
        // Informations système
        $current_dir = __DIR__;
        $parent_dir = dirname(__DIR__);
        $server_software = $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown';
        $php_version = PHP_VERSION;
        $current_user = get_current_user();
        
        // Vérifier les permissions
        $dir_perms = substr(sprintf('%o', fileperms($current_dir)), -4);
        $parent_perms = substr(sprintf('%o', fileperms($parent_dir)), -4);
        
        // Vérifier si .htaccess existe
        $htaccess_exists = file_exists('.htaccess');
        $htaccess_readable = $htaccess_exists && is_readable('.htaccess');
        
        // Vérifier les fichiers importants
        $files_to_check = [
            'login.html' => 'Page de connexion',
            'index.html' => 'Page d\'accueil',
            'index.php' => 'Fichier d\'index PHP',
            'config.php' => 'Configuration',
            '.htaccess' => 'Configuration Apache'
        ];
        ?>
        
        <div class="diagnostic-item <?php echo ($dir_perms >= '0755') ? 'success' : 'error'; ?>">
            <span class="status-icon"><?php echo ($dir_perms >= '0755') ? '✅' : '❌'; ?></span>
            <strong>Permissions du dossier app_v5 :</strong> <?php echo $dir_perms; ?>
            <?php if ($dir_perms < '0755'): ?>
                <p style="color: #DC2626; margin-top: 10px;">
                    ⚠️ Les permissions sont trop restrictives. Elles doivent être au minimum 755.
                </p>
            <?php endif; ?>
        </div>
        
        <div class="diagnostic-item <?php echo $htaccess_exists ? 'success' : 'warning'; ?>">
            <span class="status-icon"><?php echo $htaccess_exists ? '✅' : '⚠️'; ?></span>
            <strong>Fichier .htaccess :</strong> <?php echo $htaccess_exists ? 'Présent' : 'Absent'; ?>
            <?php if (!$htaccess_exists): ?>
                <p style="color: #92400E; margin-top: 10px;">
                    Le fichier .htaccess est nécessaire pour configurer les permissions Apache.
                </p>
            <?php endif; ?>
        </div>
        
        <div class="diagnostic-item">
            <strong>📊 Informations système :</strong>
            <table style="margin-top: 15px;">
                <tr>
                    <th>Paramètre</th>
                    <th>Valeur</th>
                </tr>
                <tr>
                    <td>Serveur</td>
                    <td><?php echo htmlspecialchars($server_software); ?></td>
                </tr>
                <tr>
                    <td>PHP Version</td>
                    <td><?php echo $php_version; ?></td>
                </tr>
                <tr>
                    <td>Utilisateur</td>
                    <td><?php echo htmlspecialchars($current_user); ?></td>
                </tr>
                <tr>
                    <td>Dossier actuel</td>
                    <td><?php echo htmlspecialchars($current_dir); ?></td>
                </tr>
            </table>
        </div>
        
        <div class="diagnostic-item">
            <strong>📁 Vérification des fichiers :</strong>
            <table style="margin-top: 15px;">
                <tr>
                    <th>Fichier</th>
                    <th>Statut</th>
                    <th>Permissions</th>
                </tr>
                <?php foreach ($files_to_check as $file => $description): ?>
                <tr>
                    <td><?php echo $file; ?></td>
                    <td>
                        <?php if (file_exists($file)): ?>
                            <span style="color: #059669;">✅ Existe</span>
                        <?php else: ?>
                            <span style="color: #DC2626;">❌ Manquant</span>
                        <?php endif; ?>
                    </td>
                    <td>
                        <?php 
                        if (file_exists($file)) {
                            $perms = substr(sprintf('%o', fileperms($file)), -4);
                            echo "<span class='permission-code'>$perms</span>";
                        } else {
                            echo "-";
                        }
                        ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </table>
        </div>
        
        <div class="solution-box">
            <h2 style="margin-top: 0;">🚀 Solutions pour corriger l'erreur 403</h2>
            
            <h3>Option 1 : Via cPanel File Manager</h3>
            <ol>
                <li>Connectez-vous à cPanel</li>
                <li>Ouvrez File Manager</li>
                <li>Naviguez vers <code>/public_html/app_v5/</code></li>
                <li>Clic droit sur le dossier → "Change Permissions"</li>
                <li>Définir à <strong>755</strong> (rwxr-xr-x)</li>
                <li>Cochez "Recurse into subdirectories"</li>
                <li>Cliquez "Change Permissions"</li>
            </ol>
            
            <h3>Option 2 : Créer/Modifier .htaccess</h3>
            <p>Créez un fichier <code>.htaccess</code> dans <code>/app_v5/</code> avec :</p>
            <div class="code-block">
Options +FollowSymLinks
DirectoryIndex index.html index.php login.html

&lt;RequireAll&gt;
    Require all granted
&lt;/RequireAll&gt;

RewriteEngine On
RewriteRule ^$ login.html [L]
            </div>
            
            <h3>Option 3 : Via SSH (si disponible)</h3>
            <div class="code-block">
cd /home/epsweb/public_html/app_v5
chmod 755 .
chmod 644 *.html *.php
chmod 755 uploads cache logs exports
            </div>
            
            <h3>Option 4 : Vérifier le fichier index</h3>
            <p>Assurez-vous d'avoir un de ces fichiers dans <code>/app_v5/</code> :</p>
            <ul>
                <li><code>index.html</code> - Page d'accueil HTML</li>
                <li><code>index.php</code> - Redirection PHP vers login.html</li>
            </ul>
        </div>
        
        <div class="diagnostic-item warning">
            <span class="status-icon">💡</span>
            <strong>Conseil important :</strong>
            <p>Si vous avez uploadé les fichiers .htaccess et index.php fournis, le problème devrait être résolu après avoir corrigé les permissions du dossier.</p>
        </div>
        
        <div style="text-align: center; margin-top: 40px;">
            <a href="login.html" class="btn">Tester l'accès à login.html</a>
            <a href="index.php" class="btn">Tester index.php</a>
            <a href="." class="btn">Tester le dossier</a>
        </div>
        
        <div style="margin-top: 40px; padding: 20px; background: #F3F4F6; border-radius: 12px;">
            <h3>📝 Résumé des actions à effectuer :</h3>
            <ol>
                <li>✅ Uploader le fichier <code>.htaccess</code> fourni</li>
                <li>✅ Uploader le fichier <code>index.php</code> fourni</li>
                <li>✅ Changer les permissions du dossier à 755</li>
                <li>✅ Vérifier que login.html existe bien</li>
                <li>✅ Tester l'accès</li>
            </ol>
        </div>
    </div>
</body>
</html>