<?php
/**
 * ╔══════════════════════════════════════════════════════════════╗
 * ║        CEDREPS ULTIMATE - Tableau de Bord Analytique         ║
 * ║          Visualisations & Conformité CEDREPS                 ║
 * ╚══════════════════════════════════════════════════════════════╝
 */

require_once 'config.php';
require_once 'functions.php';

if (!isAuthenticated()) {
    header('Location: login.php');
    exit;
}

$user = getCurrentUser();
$stats = getDashboardStats($user['id']);
$conformityScore = calculateCEDREPSConformity($user['id']);
?>
<!DOCTYPE html>
<html lang="fr" data-theme="light">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Tableau de Bord Analytique - CEDREPS Ultimate</title>
    
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script defer src="https://cdn.jsdelivr.net/npm/alpinejs@3.x.x/dist/cdn.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/apexcharts"></script>
    <script src="https://unpkg.com/aos@2.3.1/dist/aos.js"></script>
    <link rel="stylesheet" href="https://unpkg.com/aos@2.3.1/dist/aos.css">
    
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@100..900&family=Outfit:wght@100..900&display=swap');
        
        * { font-family: 'Inter', system-ui, sans-serif; }
        h1, h2, h3, h4, h5, h6 { font-family: 'Outfit', sans-serif; }
        
        .stat-card {
            position: relative;
            overflow: hidden;
            transition: all 0.3s ease;
        }
        
        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
            transition: left 0.5s;
        }
        
        .stat-card:hover::before {
            left: 100%;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
        }
        
        .progress-ring {
            transform: rotate(-90deg);
        }
        
        .progress-ring__circle {
            stroke-dasharray: 440;
            stroke-dashoffset: 440;
            animation: progress 2s ease-out forwards;
        }
        
        @keyframes progress {
            to {
                stroke-dashoffset: var(--progress);
            }
        }
        
        .conformity-badge {
            position: relative;
            display: inline-block;
            padding: 0.5rem 1rem;
            background: linear-gradient(135deg, var(--badge-color-1), var(--badge-color-2));
            color: white;
            border-radius: 9999px;
            font-weight: bold;
            box-shadow: 0 4px 15px rgba(0,0,0,0.2);
        }
        
        .pulse-dot {
            position: absolute;
            top: -2px;
            right: -2px;
            width: 12px;
            height: 12px;
            background: #10B981;
            border-radius: 50%;
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0% {
                box-shadow: 0 0 0 0 rgba(16, 185, 129, 0.7);
            }
            70% {
                box-shadow: 0 0 0 10px rgba(16, 185, 129, 0);
            }
            100% {
                box-shadow: 0 0 0 0 rgba(16, 185, 129, 0);
            }
        }
        
        .chart-container {
            position: relative;
            height: 300px;
            width: 100%;
        }
        
        .tooltip-custom {
            background: rgba(31, 41, 55, 0.95);
            backdrop-filter: blur(10px);
            color: white;
            padding: 8px 12px;
            border-radius: 8px;
            font-size: 12px;
            pointer-events: none;
            position: absolute;
            transform: translate(-50%, -100%);
            z-index: 1000;
        }
    </style>
</head>
<body class="min-h-screen bg-gradient-to-br from-gray-50 via-blue-50 to-purple-50" x-data="dashboard">
    
    <!-- Header Navigation -->
    <header class="bg-white/80 backdrop-blur-lg shadow-sm sticky top-0 z-40">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex items-center justify-between py-4">
                <div class="flex items-center space-x-4">
                    <h1 class="text-2xl font-bold bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">
                        Tableau de Bord Analytique
                    </h1>
                    <div class="conformity-badge" style="--badge-color-1: <?= $conformityScore >= 80 ? '#10B981' : ($conformityScore >= 60 ? '#F59E0B' : '#EF4444') ?>; --badge-color-2: <?= $conformityScore >= 80 ? '#059669' : ($conformityScore >= 60 ? '#D97706' : '#DC2626') ?>">
                        <span class="pulse-dot"></span>
                        <i class="fas fa-award mr-2"></i>
                        <?= $conformityScore ?>% Conformité CEDREPS
                    </div>
                </div>
                
                <!-- Date Range Selector -->
                <div class="flex items-center space-x-4">
                    <select class="px-4 py-2 bg-gray-100 rounded-lg text-sm focus:outline-none focus:ring-2 focus:ring-blue-500">
                        <option>Cette année</option>
                        <option>Ce trimestre</option>
                        <option>Ce mois</option>
                        <option>Cette semaine</option>
                    </select>
                    <button class="px-4 py-2 bg-gradient-to-r from-blue-500 to-purple-600 text-white rounded-lg text-sm hover:shadow-lg transition">
                        <i class="fas fa-download mr-2"></i> Exporter
                    </button>
                </div>
            </div>
        </div>
    </header>
    
    <main class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        
        <!-- Key Metrics -->
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
            <!-- Total Cycles -->
            <div class="stat-card bg-white rounded-xl shadow-sm p-6" data-aos="fade-up" data-aos-delay="100">
                <div class="flex items-center justify-between mb-4">
                    <div class="w-12 h-12 rounded-xl bg-gradient-to-r from-blue-500 to-blue-600 flex items-center justify-center text-white">
                        <i class="fas fa-sync"></i>
                    </div>
                    <span class="text-xs text-green-600 bg-green-100 px-2 py-1 rounded-full">
                        <i class="fas fa-arrow-up mr-1"></i> +12%
                    </span>
                </div>
                <h3 class="text-3xl font-bold text-gray-900"><?= $stats['total_cycles'] ?></h3>
                <p class="text-sm text-gray-600 mt-1">Cycles créés</p>
                <div class="mt-4 h-1 bg-gray-200 rounded-full overflow-hidden">
                    <div class="h-full bg-gradient-to-r from-blue-500 to-blue-600 rounded-full" style="width: 75%"></div>
                </div>
            </div>
            
            <!-- Total Lessons -->
            <div class="stat-card bg-white rounded-xl shadow-sm p-6" data-aos="fade-up" data-aos-delay="200">
                <div class="flex items-center justify-between mb-4">
                    <div class="w-12 h-12 rounded-xl bg-gradient-to-r from-purple-500 to-purple-600 flex items-center justify-center text-white">
                        <i class="fas fa-book"></i>
                    </div>
                    <span class="text-xs text-green-600 bg-green-100 px-2 py-1 rounded-full">
                        <i class="fas fa-arrow-up mr-1"></i> +8%
                    </span>
                </div>
                <h3 class="text-3xl font-bold text-gray-900"><?= $stats['total_lessons'] ?></h3>
                <p class="text-sm text-gray-600 mt-1">Leçons dispensées</p>
                <div class="mt-4 h-1 bg-gray-200 rounded-full overflow-hidden">
                    <div class="h-full bg-gradient-to-r from-purple-500 to-purple-600 rounded-full" style="width: 82%"></div>
                </div>
            </div>
            
            <!-- Students Progress -->
            <div class="stat-card bg-white rounded-xl shadow-sm p-6" data-aos="fade-up" data-aos-delay="300">
                <div class="flex items-center justify-between mb-4">
                    <div class="w-12 h-12 rounded-xl bg-gradient-to-r from-green-500 to-green-600 flex items-center justify-center text-white">
                        <i class="fas fa-users"></i>
                    </div>
                    <span class="text-xs text-green-600 bg-green-100 px-2 py-1 rounded-full">
                        <i class="fas fa-arrow-up mr-1"></i> +15%
                    </span>
                </div>
                <h3 class="text-3xl font-bold text-gray-900"><?= $stats['total_students'] ?></h3>
                <p class="text-sm text-gray-600 mt-1">Élèves en progression</p>
                <div class="mt-4 h-1 bg-gray-200 rounded-full overflow-hidden">
                    <div class="h-full bg-gradient-to-r from-green-500 to-green-600 rounded-full" style="width: 89%"></div>
                </div>
            </div>
            
            <!-- FPS Created -->
            <div class="stat-card bg-white rounded-xl shadow-sm p-6" data-aos="fade-up" data-aos-delay="400">
                <div class="flex items-center justify-between mb-4">
                    <div class="w-12 h-12 rounded-xl bg-gradient-to-r from-orange-500 to-orange-600 flex items-center justify-center text-white">
                        <i class="fas fa-cogs"></i>
                    </div>
                    <span class="text-xs text-green-600 bg-green-100 px-2 py-1 rounded-full">
                        <i class="fas fa-arrow-up mr-1"></i> +20%
                    </span>
                </div>
                <h3 class="text-3xl font-bold text-gray-900"><?= $stats['total_fps'] ?></h3>
                <p class="text-sm text-gray-600 mt-1">FPS créées</p>
                <div class="mt-4 h-1 bg-gray-200 rounded-full overflow-hidden">
                    <div class="h-full bg-gradient-to-r from-orange-500 to-orange-600 rounded-full" style="width: 65%"></div>
                </div>
            </div>
        </div>
        
        <!-- ECC Coverage & Conformity Score -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-8 mb-8">
            <!-- ECC Coverage Radar -->
            <div class="bg-white rounded-xl shadow-sm p-6" data-aos="fade-right">
                <h2 class="text-lg font-bold mb-4">Couverture des 6 ECC</h2>
                <div class="relative">
                    <canvas id="eccRadarChart" class="chart-container"></canvas>
                    <div class="mt-4 grid grid-cols-3 gap-4">
                        <div class="text-center">
                            <div class="text-2xl font-bold text-green-600">3/6</div>
                            <div class="text-xs text-gray-500">ECC maîtrisées</div>
                        </div>
                        <div class="text-center">
                            <div class="text-2xl font-bold text-blue-600">2/6</div>
                            <div class="text-xs text-gray-500">En progression</div>
                        </div>
                        <div class="text-center">
                            <div class="text-2xl font-bold text-gray-400">1/6</div>
                            <div class="text-xs text-gray-500">À explorer</div>
                        </div>
                    </div>
                </div>
                
                <!-- ECC Detail List -->
                <div class="mt-6 space-y-3">
                    <div class="flex items-center justify-between p-3 bg-red-50 rounded-lg">
                        <div class="flex items-center space-x-3">
                            <div class="w-8 h-8 rounded-lg bg-red-500 flex items-center justify-center text-white text-sm">
                                <i class="fas fa-trophy"></i>
                            </div>
                            <div>
                                <p class="font-semibold text-sm">ECC1 - Performance</p>
                                <p class="text-xs text-gray-500">3 cycles • 24 leçons</p>
                            </div>
                        </div>
                        <div class="flex items-center">
                            <div class="w-20 bg-gray-200 rounded-full h-2 mr-2">
                                <div class="bg-red-500 h-2 rounded-full" style="width: 85%"></div>
                            </div>
                            <span class="text-xs font-semibold text-red-600">85%</span>
                        </div>
                    </div>
                    
                    <div class="flex items-center justify-between p-3 bg-cyan-50 rounded-lg">
                        <div class="flex items-center space-x-3">
                            <div class="w-8 h-8 rounded-lg bg-cyan-500 flex items-center justify-center text-white text-sm">
                                <i class="fas fa-compass"></i>
                            </div>
                            <div>
                                <p class="font-semibold text-sm">ECC2 - Aventure</p>
                                <p class="text-xs text-gray-500">2 cycles • 16 leçons</p>
                            </div>
                        </div>
                        <div class="flex items-center">
                            <div class="w-20 bg-gray-200 rounded-full h-2 mr-2">
                                <div class="bg-cyan-500 h-2 rounded-full" style="width: 60%"></div>
                            </div>
                            <span class="text-xs font-semibold text-cyan-600">60%</span>
                        </div>
                    </div>
                    
                    <div class="flex items-center justify-between p-3 bg-yellow-50 rounded-lg">
                        <div class="flex items-center space-x-3">
                            <div class="w-8 h-8 rounded-lg bg-yellow-500 flex items-center justify-center text-white text-sm">
                                <i class="fas fa-handshake"></i>
                            </div>
                            <div>
                                <p class="font-semibold text-sm">ECC4 - Confrontation</p>
                                <p class="text-xs text-gray-500">4 cycles • 32 leçons</p>
                            </div>
                        </div>
                        <div class="flex items-center">
                            <div class="w-20 bg-gray-200 rounded-full h-2 mr-2">
                                <div class="bg-yellow-500 h-2 rounded-full" style="width: 92%"></div>
                            </div>
                            <span class="text-xs font-semibold text-yellow-600">92%</span>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- CEDREPS Conformity Score Breakdown -->
            <div class="bg-white rounded-xl shadow-sm p-6" data-aos="fade-left">
                <h2 class="text-lg font-bold mb-4">Score de Conformité CEDREPS</h2>
                
                <!-- Overall Score Circle -->
                <div class="flex justify-center mb-6">
                    <div class="relative w-48 h-48">
                        <svg class="progress-ring w-48 h-48">
                            <circle cx="96" cy="96" r="70" stroke="#E5E7EB" stroke-width="12" fill="none" />
                            <circle class="progress-ring__circle" cx="96" cy="96" r="70" 
                                    stroke="url(#gradient)" stroke-width="12" fill="none"
                                    style="--progress: <?= 440 - (440 * $conformityScore / 100) ?>px" />
                            <defs>
                                <linearGradient id="gradient">
                                    <stop offset="0%" stop-color="#3B82F6" />
                                    <stop offset="100%" stop-color="#8B5CF6" />
                                </linearGradient>
                            </defs>
                        </svg>
                        <div class="absolute inset-0 flex items-center justify-center">
                            <div class="text-center">
                                <p class="text-4xl font-bold"><?= $conformityScore ?>%</p>
                                <p class="text-sm text-gray-500">Conformité</p>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Detailed Criteria -->
                <div class="space-y-3">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center space-x-2">
                            <i class="fas fa-check-circle text-green-500"></i>
                            <span class="text-sm">Objets d'Enseignement ciblés</span>
                        </div>
                        <span class="text-sm font-semibold text-green-600">100%</span>
                    </div>
                    
                    <div class="flex items-center justify-between">
                        <div class="flex items-center space-x-2">
                            <i class="fas fa-check-circle text-green-500"></i>
                            <span class="text-sm">FPS avec contraintes emblématiques</span>
                        </div>
                        <span class="text-sm font-semibold text-green-600">95%</span>
                    </div>
                    
                    <div class="flex items-center justify-between">
                        <div class="flex items-center space-x-2">
                            <i class="fas fa-exclamation-circle text-yellow-500"></i>
                            <span class="text-sm">Évaluation des 4 dimensions</span>
                        </div>
                        <span class="text-sm font-semibold text-yellow-600">75%</span>
                    </div>
                    
                    <div class="flex items-center justify-between">
                        <div class="flex items-center space-x-2">
                            <i class="fas fa-times-circle text-red-500"></i>
                            <span class="text-sm">Couverture des 6 ECC</span>
                        </div>
                        <span class="text-sm font-semibold text-red-600">50%</span>
                    </div>
                    
                    <div class="flex items-center justify-between">
                        <div class="flex items-center space-x-2">
                            <i class="fas fa-check-circle text-green-500"></i>
                            <span class="text-sm">Double boucle appliquée</span>
                        </div>
                        <span class="text-sm font-semibold text-green-600">88%</span>
                    </div>
                </div>
                
                <!-- Recommendations -->
                <div class="mt-6 p-4 bg-blue-50 rounded-lg">
                    <h3 class="font-semibold text-sm text-blue-900 mb-2">
                        <i class="fas fa-lightbulb mr-2"></i>Recommandations pour améliorer
                    </h3>
                    <ul class="text-xs text-blue-800 space-y-1">
                        <li>• Créer au moins 1 cycle pour les ECC manquantes (ECC3, ECC5, ECC6)</li>
                        <li>• Systématiser l'évaluation des 4 dimensions dans toutes les leçons</li>
                        <li>• Documenter les contraintes emblématiques de vos FPS</li>
                    </ul>
                </div>
            </div>
        </div>
        
        <!-- Students Progress & Evaluation Dimensions -->
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-8 mb-8">
            <!-- Students Progress Over Time -->
            <div class="lg:col-span-2 bg-white rounded-xl shadow-sm p-6" data-aos="fade-up">
                <div class="flex items-center justify-between mb-4">
                    <h2 class="text-lg font-bold">Progression des Élèves</h2>
                    <select class="px-3 py-1 text-sm bg-gray-100 rounded-lg focus:outline-none">
                        <option>Toutes les classes</option>
                        <option>6ème A</option>
                        <option>5ème A</option>
                    </select>
                </div>
                <div id="progressChart" class="chart-container"></div>
                
                <!-- Legend -->
                <div class="mt-4 flex items-center justify-center space-x-6">
                    <div class="flex items-center">
                        <div class="w-3 h-3 rounded-full bg-red-500 mr-2"></div>
                        <span class="text-xs text-gray-600">Motricité</span>
                    </div>
                    <div class="flex items-center">
                        <div class="w-3 h-3 rounded-full bg-blue-500 mr-2"></div>
                        <span class="text-xs text-gray-600">Intelligibilité</span>
                    </div>
                    <div class="flex items-center">
                        <div class="w-3 h-3 rounded-full bg-yellow-500 mr-2"></div>
                        <span class="text-xs text-gray-600">Sensibilité</span>
                    </div>
                    <div class="flex items-center">
                        <div class="w-3 h-3 rounded-full bg-green-500 mr-2"></div>
                        <span class="text-xs text-gray-600">Sociabilité</span>
                    </div>
                </div>
            </div>
            
            <!-- Top Performing Students -->
            <div class="bg-white rounded-xl shadow-sm p-6" data-aos="fade-up">
                <h2 class="text-lg font-bold mb-4">Top Élèves</h2>
                <div class="space-y-3">
                    <div class="flex items-center space-x-3">
                        <div class="relative">
                            <img src="https://ui-avatars.com/api/?name=Lucas+Martin&background=FFD700&color=ffffff" 
                                 class="w-10 h-10 rounded-full" alt="Avatar">
                            <div class="absolute -top-1 -right-1 w-5 h-5 bg-yellow-500 rounded-full flex items-center justify-center text-white text-xs font-bold">
                                1
                            </div>
                        </div>
                        <div class="flex-1">
                            <p class="font-semibold text-sm">Lucas Martin</p>
                            <p class="text-xs text-gray-500">6ème A</p>
                        </div>
                        <div class="text-right">
                            <p class="font-bold text-sm">92%</p>
                            <p class="text-xs text-green-600">+5%</p>
                        </div>
                    </div>
                    
                    <div class="flex items-center space-x-3">
                        <div class="relative">
                            <img src="https://ui-avatars.com/api/?name=Emma+Dubois&background=C0C0C0&color=ffffff" 
                                 class="w-10 h-10 rounded-full" alt="Avatar">
                            <div class="absolute -top-1 -right-1 w-5 h-5 bg-gray-500 rounded-full flex items-center justify-center text-white text-xs font-bold">
                                2
                            </div>
                        </div>
                        <div class="flex-1">
                            <p class="font-semibold text-sm">Emma Dubois</p>
                            <p class="text-xs text-gray-500">5ème A</p>
                        </div>
                        <div class="text-right">
                            <p class="font-bold text-sm">89%</p>
                            <p class="text-xs text-green-600">+8%</p>
                        </div>
                    </div>
                    
                    <div class="flex items-center space-x-3">
                        <div class="relative">
                            <img src="https://ui-avatars.com/api/?name=Noah+Leroy&background=CD7F32&color=ffffff" 
                                 class="w-10 h-10 rounded-full" alt="Avatar">
                            <div class="absolute -top-1 -right-1 w-5 h-5 bg-orange-600 rounded-full flex items-center justify-center text-white text-xs font-bold">
                                3
                            </div>
                        </div>
                        <div class="flex-1">
                            <p class="font-semibold text-sm">Noah Leroy</p>
                            <p class="text-xs text-gray-500">6ème A</p>
                        </div>
                        <div class="text-right">
                            <p class="font-bold text-sm">87%</p>
                            <p class="text-xs text-green-600">+3%</p>
                        </div>
                    </div>
                </div>
                
                <!-- View All Button -->
                <button class="w-full mt-4 px-4 py-2 bg-gray-100 text-gray-700 rounded-lg text-sm hover:bg-gray-200 transition">
                    Voir tous les élèves
                </button>
            </div>
        </div>
        
        <!-- Recent Activities & Quick Actions -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-8">
            <!-- Recent Activities Timeline -->
            <div class="bg-white rounded-xl shadow-sm p-6" data-aos="fade-right">
                <h2 class="text-lg font-bold mb-4">Activités Récentes</h2>
                <div class="relative">
                    <div class="absolute left-4 top-0 bottom-0 w-0.5 bg-gray-200"></div>
                    <div class="space-y-4">
                        <div class="relative flex items-start space-x-4">
                            <div class="w-8 h-8 rounded-full bg-green-500 flex items-center justify-center text-white text-xs z-10">
                                <i class="fas fa-check"></i>
                            </div>
                            <div class="flex-1 bg-green-50 rounded-lg p-3">
                                <p class="text-sm font-semibold">Cycle complété</p>
                                <p class="text-xs text-gray-600">Handball - 6ème A terminé avec succès</p>
                                <p class="text-xs text-gray-400 mt-1">Il y a 2 heures</p>
                            </div>
                        </div>
                        
                        <div class="relative flex items-start space-x-4">
                            <div class="w-8 h-8 rounded-full bg-blue-500 flex items-center justify-center text-white text-xs z-10">
                                <i class="fas fa-plus"></i>
                            </div>
                            <div class="flex-1 bg-blue-50 rounded-lg p-3">
                                <p class="text-sm font-semibold">Nouvelle FPS créée</p>
                                <p class="text-xs text-gray-600">"Le jeu en mouvement" ajoutée à la bibliothèque</p>
                                <p class="text-xs text-gray-400 mt-1">Il y a 5 heures</p>
                            </div>
                        </div>
                        
                        <div class="relative flex items-start space-x-4">
                            <div class="w-8 h-8 rounded-full bg-purple-500 flex items-center justify-center text-white text-xs z-10">
                                <i class="fas fa-chart-line"></i>
                            </div>
                            <div class="flex-1 bg-purple-50 rounded-lg p-3">
                                <p class="text-sm font-semibold">Évaluations saisies</p>
                                <p class="text-xs text-gray-600">24 élèves évalués pour la leçon 8</p>
                                <p class="text-xs text-gray-400 mt-1">Hier</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Quick Actions -->
            <div class="bg-white rounded-xl shadow-sm p-6" data-aos="fade-left">
                <h2 class="text-lg font-bold mb-4">Actions Rapides</h2>
                <div class="grid grid-cols-2 gap-4">
                    <button class="p-4 bg-gradient-to-r from-blue-500 to-blue-600 text-white rounded-lg hover:shadow-lg transition group">
                        <i class="fas fa-plus text-2xl mb-2 group-hover:scale-110 transition"></i>
                        <p class="text-sm font-semibold">Nouveau Cycle</p>
                    </button>
                    
                    <button class="p-4 bg-gradient-to-r from-purple-500 to-purple-600 text-white rounded-lg hover:shadow-lg transition group">
                        <i class="fas fa-magic text-2xl mb-2 group-hover:scale-110 transition"></i>
                        <p class="text-sm font-semibold">Créer FPS</p>
                    </button>
                    
                    <button class="p-4 bg-gradient-to-r from-green-500 to-green-600 text-white rounded-lg hover:shadow-lg transition group">
                        <i class="fas fa-chart-bar text-2xl mb-2 group-hover:scale-110 transition"></i>
                        <p class="text-sm font-semibold">Évaluer</p>
                    </button>
                    
                    <button class="p-4 bg-gradient-to-r from-orange-500 to-orange-600 text-white rounded-lg hover:shadow-lg transition group">
                        <i class="fas fa-share-alt text-2xl mb-2 group-hover:scale-110 transition"></i>
                        <p class="text-sm font-semibold">Partager</p>
                    </button>
                </div>
                
                <!-- Upcoming Tasks -->
                <div class="mt-6 p-4 bg-yellow-50 rounded-lg">
                    <h3 class="font-semibold text-sm text-yellow-900 mb-3">
                        <i class="fas fa-clock mr-2"></i>À faire cette semaine
                    </h3>
                    <div class="space-y-2">
                        <label class="flex items-center space-x-3 cursor-pointer">
                            <input type="checkbox" class="rounded text-yellow-600">
                            <span class="text-sm text-gray-700">Créer le cycle Escalade pour les 5ème</span>
                        </label>
                        <label class="flex items-center space-x-3 cursor-pointer">
                            <input type="checkbox" class="rounded text-yellow-600">
                            <span class="text-sm text-gray-700">Évaluer la leçon 9 du cycle Handball</span>
                        </label>
                        <label class="flex items-center space-x-3 cursor-pointer">
                            <input type="checkbox" class="rounded text-yellow-600">
                            <span class="text-sm text-gray-700">Partager le bilan avec l'inspectrice</span>
                        </label>
                    </div>
                </div>
            </div>
        </div>
    </main>
    
    <!-- JavaScript -->
    <script>
        // Initialize AOS
        AOS.init({ duration: 800, once: true });
        
        // Alpine.js Component
        document.addEventListener('alpine:init', () => {
            Alpine.data('dashboard', () => ({
                init() {
                    this.initCharts();
                },
                
                initCharts() {
                    // ECC Radar Chart
                    const eccRadarCtx = document.getElementById('eccRadarChart').getContext('2d');
                    new Chart(eccRadarCtx, {
                        type: 'radar',
                        data: {
                            labels: ['ECC1 Performance', 'ECC2 Aventure', 'ECC3 Arts', 'ECC4 Confrontation', 'ECC5 Soin de soi', 'ECC6 Acrobatie'],
                            datasets: [{
                                label: 'Couverture actuelle',
                                data: [85, 60, 10, 92, 15, 20],
                                backgroundColor: 'rgba(59, 130, 246, 0.2)',
                                borderColor: 'rgba(59, 130, 246, 1)',
                                pointBackgroundColor: 'rgba(59, 130, 246, 1)',
                                pointBorderColor: '#fff',
                                pointHoverBackgroundColor: '#fff',
                                pointHoverBorderColor: 'rgba(59, 130, 246, 1)'
                            }, {
                                label: 'Objectif',
                                data: [80, 80, 80, 80, 80, 80],
                                backgroundColor: 'rgba(239, 68, 68, 0.1)',
                                borderColor: 'rgba(239, 68, 68, 0.5)',
                                borderDash: [5, 5],
                                pointBackgroundColor: 'rgba(239, 68, 68, 0.5)',
                                pointBorderColor: '#fff',
                                pointHoverBackgroundColor: '#fff',
                                pointHoverBorderColor: 'rgba(239, 68, 68, 0.5)'
                            }]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            scales: {
                                r: {
                                    beginAtZero: true,
                                    max: 100,
                                    ticks: {
                                        stepSize: 20
                                    }
                                }
                            },
                            plugins: {
                                legend: {
                                    position: 'bottom'
                                }
                            }
                        }
                    });
                    
                    // Progress Line Chart
                    const progressOptions = {
                        series: [
                            {
                                name: 'Motricité',
                                data: [65, 68, 70, 72, 75, 78, 82]
                            },
                            {
                                name: 'Intelligibilité',
                                data: [60, 62, 65, 68, 71, 74, 78]
                            },
                            {
                                name: 'Sensibilité',
                                data: [55, 58, 60, 63, 66, 69, 72]
                            },
                            {
                                name: 'Sociabilité',
                                data: [70, 72, 74, 76, 78, 80, 83]
                            }
                        ],
                        chart: {
                            height: 300,
                            type: 'line',
                            toolbar: {
                                show: false
                            },
                            animations: {
                                enabled: true,
                                easing: 'easeinout',
                                speed: 800
                            }
                        },
                        colors: ['#EF4444', '#3B82F6', '#F59E0B', '#10B981'],
                        stroke: {
                            curve: 'smooth',
                            width: 3
                        },
                        xaxis: {
                            categories: ['Sept', 'Oct', 'Nov', 'Déc', 'Jan', 'Fév', 'Mars']
                        },
                        yaxis: {
                            min: 50,
                            max: 100
                        },
                        grid: {
                            borderColor: '#f1f1f1'
                        },
                        markers: {
                            size: 4,
                            colors: ['#EF4444', '#3B82F6', '#F59E0B', '#10B981'],
                            strokeColors: '#fff',
                            strokeWidth: 2,
                            hover: {
                                size: 6
                            }
                        },
                        tooltip: {
                            theme: 'light',
                            y: {
                                formatter: function(val) {
                                    return val + "%";
                                }
                            }
                        }
                    };
                    
                    const progressChart = new ApexCharts(document.querySelector("#progressChart"), progressOptions);
                    progressChart.render();
                }
            }));
        });
    </script>
</body>
</html>