-- ╔══════════════════════════════════════════════════════════════╗
-- ║       CEDREPS ULTIMATE - Structure Base de Données v2.0      ║
-- ║                   Architecture Complète                       ║
-- ╚══════════════════════════════════════════════════════════════╝

SET SQL_MODE = "NO_AUTO_VALUE_ON_ZERO";
START TRANSACTION;
SET time_zone = "+00:00";
SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci;

-- ===================================================================
-- SUPPRESSION DES TABLES EXISTANTES (SI NÉCESSAIRE)
-- ===================================================================

DROP TABLE IF EXISTS `share_permissions`;
DROP TABLE IF EXISTS `cycle_shares`;
DROP TABLE IF EXISTS `fps_library_likes`;
DROP TABLE IF EXISTS `fps_library_downloads`;
DROP TABLE IF EXISTS `fps_library_comments`;
DROP TABLE IF EXISTS `fps_library`;
DROP TABLE IF EXISTS `learning_progress`;
DROP TABLE IF EXISTS `notifications`;
DROP TABLE IF EXISTS `activity_logs`;
DROP TABLE IF EXISTS `evaluation_dimensions`;
DROP TABLE IF EXISTS `evaluations`;
DROP TABLE IF EXISTS `student_roles`;
DROP TABLE IF EXISTS `lesson_resources`;
DROP TABLE IF EXISTS `lessons`;
DROP TABLE IF EXISTS `cycle_oe_mappings`;
DROP TABLE IF EXISTS `objets_enseignement`;
DROP TABLE IF EXISTS `cycles`;
DROP TABLE IF EXISTS `students`;
DROP TABLE IF EXISTS `class_teachers`;
DROP TABLE IF EXISTS `classes`;
DROP TABLE IF EXISTS `user_achievements`;
DROP TABLE IF EXISTS `achievements`;
DROP TABLE IF EXISTS `user_roles`;
DROP TABLE IF EXISTS `role_permissions`;
DROP TABLE IF EXISTS `permissions`;
DROP TABLE IF EXISTS `roles`;
DROP TABLE IF EXISTS `password_resets`;
DROP TABLE IF EXISTS `login_attempts`;
DROP TABLE IF EXISTS `users`;
DROP TABLE IF EXISTS `ecc_definitions`;
DROP TABLE IF EXISTS `ppsad_definitions`;

-- ===================================================================
-- TABLES DE RÉFÉRENCE CEDREPS
-- ===================================================================

-- Table des définitions ECC (Expériences Corporelles et Culturelles)
CREATE TABLE `ecc_definitions` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `code` VARCHAR(10) NOT NULL UNIQUE,
    `name` VARCHAR(100) NOT NULL,
    `icon` VARCHAR(50),
    `color` VARCHAR(7),
    `gradient` TEXT,
    `description` TEXT,
    `imaginaire_symbolique` TEXT,
    `but_activite` TEXT,
    `mobiles_typiques` TEXT,
    `ppsad_examples` JSON,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_code (code)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table des définitions PPSAD
CREATE TABLE `ppsad_definitions` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `name` VARCHAR(100) NOT NULL,
    `code` VARCHAR(50),
    `ecc_type` VARCHAR(10),
    `category` VARCHAR(100),
    `description` TEXT,
    `fonds_culturel_mobiles` TEXT,
    `fonds_culturel_principes` TEXT,
    `fonds_culturel_rapport_autre` TEXT,
    `icon` VARCHAR(50),
    `color` VARCHAR(7),
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (ecc_type) REFERENCES ecc_definitions(code) ON DELETE SET NULL,
    INDEX idx_ecc_type (ecc_type),
    INDEX idx_name (name)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================================================
-- GESTION DES UTILISATEURS ET RÔLES
-- ===================================================================

-- Table des utilisateurs
CREATE TABLE `users` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `name` VARCHAR(255) NOT NULL,
    `email` VARCHAR(255) NOT NULL UNIQUE,
    `password` VARCHAR(255) NOT NULL,
    `avatar_url` VARCHAR(500),
    `bio` TEXT,
    `establishment` VARCHAR(255),
    `academy` VARCHAR(100),
    `cedreps_level` INT DEFAULT 1,
    `cedreps_xp` INT DEFAULT 0,
    `email_verified_at` TIMESTAMP NULL,
    `remember_token` VARCHAR(100),
    `two_factor_secret` TEXT,
    `two_factor_recovery_codes` TEXT,
    `last_login_at` TIMESTAMP NULL,
    `is_active` BOOLEAN DEFAULT TRUE,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_email (email),
    INDEX idx_cedreps_level (cedreps_level)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table des rôles système
CREATE TABLE `roles` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `code` VARCHAR(50) NOT NULL UNIQUE,
    `name` VARCHAR(100) NOT NULL,
    `level` INT NOT NULL DEFAULT 0,
    `icon` VARCHAR(50),
    `color` VARCHAR(7),
    `description` TEXT,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_code (code),
    INDEX idx_level (level)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table des permissions
CREATE TABLE `permissions` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `code` VARCHAR(100) NOT NULL UNIQUE,
    `name` VARCHAR(255) NOT NULL,
    `category` VARCHAR(100),
    `description` TEXT,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_code (code),
    INDEX idx_category (category)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table de liaison rôles-permissions
CREATE TABLE `role_permissions` (
    `role_id` INT NOT NULL,
    `permission_id` INT NOT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (role_id, permission_id),
    FOREIGN KEY (role_id) REFERENCES roles(id) ON DELETE CASCADE,
    FOREIGN KEY (permission_id) REFERENCES permissions(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table de liaison utilisateurs-rôles
CREATE TABLE `user_roles` (
    `user_id` INT NOT NULL,
    `role_id` INT NOT NULL,
    `assigned_by` INT,
    `assigned_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `expires_at` TIMESTAMP NULL,
    PRIMARY KEY (user_id, role_id),
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (role_id) REFERENCES roles(id) ON DELETE CASCADE,
    FOREIGN KEY (assigned_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_user_id (user_id),
    INDEX idx_role_id (role_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================================================
-- GESTION DES CLASSES ET ÉLÈVES
-- ===================================================================

-- Table des classes
CREATE TABLE `classes` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `name` VARCHAR(255) NOT NULL,
    `level` VARCHAR(255),
    `school_year` VARCHAR(20),
    `teacher_id` INT NOT NULL,
    `description` TEXT,
    `student_count` INT DEFAULT 0,
    `is_active` BOOLEAN DEFAULT TRUE,
    `metadata` JSON,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (teacher_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_teacher_id (teacher_id),
    INDEX idx_school_year (school_year)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table de liaison classes-enseignants (multi-enseignants)
CREATE TABLE `class_teachers` (
    `class_id` INT NOT NULL,
    `user_id` INT NOT NULL,
    `role` ENUM('main', 'assistant', 'observer') DEFAULT 'assistant',
    `can_edit` BOOLEAN DEFAULT FALSE,
    `assigned_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (class_id, user_id),
    FOREIGN KEY (class_id) REFERENCES classes(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_class_id (class_id),
    INDEX idx_user_id (user_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table des élèves
CREATE TABLE `students` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `class_id` INT NOT NULL,
    `user_id` INT UNIQUE,
    `name` VARCHAR(255) NOT NULL,
    `date_of_birth` DATE,
    `gender` ENUM('M', 'F', 'other'),
    `level` ENUM('debutant', 'intermediaire', 'avance', 'expert') DEFAULT 'debutant',
    `comments` TEXT,
    `dispense` BOOLEAN DEFAULT FALSE,
    `dispense_reason` TEXT,
    `medical_notes` TEXT,
    `avatar_url` VARCHAR(500),
    `metadata` JSON,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (class_id) REFERENCES classes(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_class_id (class_id),
    INDEX idx_name (name)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================================================
-- GESTION DES CYCLES ET OBJETS D'ENSEIGNEMENT
-- ===================================================================

-- Table des objets d'enseignement référentiels
CREATE TABLE `objets_enseignement` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `ppsad` VARCHAR(100) NOT NULL,
    `ecc_type` VARCHAR(10),
    `niveau_scolaire` ENUM('cycle3', 'cycle4', 'lycee') NOT NULL,
    `objet_enseignement` TEXT NOT NULL,
    `pas_en_avant` TEXT,
    `prerequis` TEXT,
    `exemples_fps` TEXT,
    `ordre_progression` INT,
    `source_cahier` VARCHAR(100),
    `validated_cedreps` BOOLEAN DEFAULT FALSE,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (ecc_type) REFERENCES ecc_definitions(code) ON DELETE SET NULL,
    INDEX idx_ppsad_niveau (ppsad, niveau_scolaire),
    INDEX idx_ecc_type (ecc_type)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table des cycles
CREATE TABLE `cycles` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `class_id` INT NOT NULL,
    `name` VARCHAR(255) NOT NULL,
    `ppsad` VARCHAR(255) NOT NULL,
    `ecc_type` VARCHAR(10),
    `objective` TEXT NOT NULL,
    `fps_name` VARCHAR(255),
    `fps_description` TEXT,
    `contraintes_emblematiques` TEXT,
    `fonds_culturel_mobiles` TEXT,
    `fonds_culturel_principes` TEXT,
    `fonds_culturel_rapport_autre` TEXT,
    `start_date` DATE,
    `end_date` DATE,
    `nb_lessons_planned` INT DEFAULT 10,
    `nb_lessons_completed` INT DEFAULT 0,
    `status` ENUM('draft', 'active', 'completed', 'archived') DEFAULT 'draft',
    `conformity_score` INT DEFAULT 0,
    `metadata` JSON,
    `created_by` INT,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (class_id) REFERENCES classes(id) ON DELETE CASCADE,
    FOREIGN KEY (ecc_type) REFERENCES ecc_definitions(code) ON DELETE SET NULL,
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_class_id (class_id),
    INDEX idx_status (status),
    INDEX idx_ecc_type (ecc_type),
    INDEX idx_dates (start_date, end_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table de liaison cycles-objets d'enseignement
CREATE TABLE `cycle_oe_mappings` (
    `cycle_id` INT NOT NULL,
    `oe_id` INT NOT NULL,
    `is_primary` BOOLEAN DEFAULT FALSE,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (cycle_id, oe_id),
    FOREIGN KEY (cycle_id) REFERENCES cycles(id) ON DELETE CASCADE,
    FOREIGN KEY (oe_id) REFERENCES objets_enseignement(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================================================
-- GESTION DES LEÇONS
-- ===================================================================

-- Table des leçons
CREATE TABLE `lessons` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `cycle_id` INT NOT NULL,
    `lesson_number` INT NOT NULL,
    `lesson_date` DATE NOT NULL,
    `title` VARCHAR(255) NOT NULL,
    `fps` TEXT,
    `cp` TEXT COMMENT 'Contenus Prioritaires',
    `description` TEXT,
    `objectives` TEXT,
    `double_boucle_type` ENUM('grande_boucle', 'petite_boucle', 'mixte') DEFAULT 'grande_boucle',
    `material` TEXT,
    `duration_minutes` INT DEFAULT 55,
    `location` VARCHAR(255),
    `status` ENUM('planned', 'in_progress', 'completed', 'cancelled') DEFAULT 'planned',
    `notes` TEXT,
    `metadata` JSON,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (cycle_id) REFERENCES cycles(id) ON DELETE CASCADE,
    INDEX idx_cycle_id (cycle_id),
    INDEX idx_lesson_date (lesson_date),
    INDEX idx_status (status),
    UNIQUE KEY unique_cycle_number (cycle_id, lesson_number)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table des ressources de leçon
CREATE TABLE `lesson_resources` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `lesson_id` INT NOT NULL,
    `type` ENUM('document', 'video', 'image', 'link', 'other') NOT NULL,
    `name` VARCHAR(255) NOT NULL,
    `url` VARCHAR(500),
    `file_path` VARCHAR(500),
    `description` TEXT,
    `order_index` INT DEFAULT 0,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (lesson_id) REFERENCES lessons(id) ON DELETE CASCADE,
    INDEX idx_lesson_id (lesson_id),
    INDEX idx_type (type)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================================================
-- GESTION DES ÉVALUATIONS
-- ===================================================================

-- Table des évaluations
CREATE TABLE `evaluations` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `lesson_id` INT NOT NULL,
    `student_id` INT NOT NULL,
    `evaluation_date` DATE,
    `global_score` DECIMAL(5,2),
    `motricite` DECIMAL(5,2),
    `intelligibilite` DECIMAL(5,2),
    `sensibilite` DECIMAL(5,2),
    `sociabilite` DECIMAL(5,2),
    `progress_indicator` ENUM('en_progres', 'acquis', 'expert') DEFAULT 'en_progres',
    `social_role` VARCHAR(100),
    `remarks` TEXT,
    `peer_evaluation` JSON,
    `self_evaluation` JSON,
    `metadata` JSON,
    `evaluated_by` INT,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (lesson_id) REFERENCES lessons(id) ON DELETE CASCADE,
    FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
    FOREIGN KEY (evaluated_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_lesson_student (lesson_id, student_id),
    INDEX idx_student_id (student_id),
    INDEX idx_evaluation_date (evaluation_date),
    UNIQUE KEY unique_lesson_student (lesson_id, student_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table des dimensions d'évaluation détaillées
CREATE TABLE `evaluation_dimensions` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `evaluation_id` INT NOT NULL,
    `dimension` VARCHAR(50) NOT NULL,
    `criterion` VARCHAR(255) NOT NULL,
    `score` DECIMAL(5,2),
    `comment` TEXT,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (evaluation_id) REFERENCES evaluations(id) ON DELETE CASCADE,
    INDEX idx_evaluation_id (evaluation_id),
    INDEX idx_dimension (dimension)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table des rôles sociaux des élèves
CREATE TABLE `student_roles` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `name` VARCHAR(100) NOT NULL UNIQUE,
    `description` TEXT,
    `icon` VARCHAR(50),
    `color` VARCHAR(7),
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================================================
-- BIBLIOTHÈQUE DE FPS ET PARTAGE
-- ===================================================================

-- Table de la bibliothèque de FPS
CREATE TABLE `fps_library` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `name` VARCHAR(255) NOT NULL,
    `ppsad` VARCHAR(100) NOT NULL,
    `ecc_type` VARCHAR(10),
    `niveau_scolaire` ENUM('cycle3', 'cycle4', 'lycee') NOT NULL,
    `objet_enseignement` TEXT,
    `contraintes_emblematiques` TEXT,
    `description` TEXT,
    `fps_details` JSON,
    `cp_suggestions` TEXT,
    `double_boucle_structure` JSON,
    `author_id` INT,
    `establishment` VARCHAR(255),
    `academy` VARCHAR(100),
    `validated_cedreps` BOOLEAN DEFAULT FALSE,
    `validation_date` TIMESTAMP NULL,
    `validated_by` INT,
    `likes_count` INT DEFAULT 0,
    `downloads_count` INT DEFAULT 0,
    `views_count` INT DEFAULT 0,
    `is_public` BOOLEAN DEFAULT TRUE,
    `tags` JSON,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (author_id) REFERENCES users(id) ON DELETE SET NULL,
    FOREIGN KEY (ecc_type) REFERENCES ecc_definitions(code) ON DELETE SET NULL,
    FOREIGN KEY (validated_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_ppsad_niveau (ppsad, niveau_scolaire),
    INDEX idx_validated (validated_cedreps),
    INDEX idx_author (author_id),
    INDEX idx_public (is_public),
    FULLTEXT idx_search (name, description, objet_enseignement)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table des likes sur les FPS
CREATE TABLE `fps_library_likes` (
    `user_id` INT NOT NULL,
    `fps_id` INT NOT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (user_id, fps_id),
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (fps_id) REFERENCES fps_library(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table des téléchargements de FPS
CREATE TABLE `fps_library_downloads` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `fps_id` INT NOT NULL,
    `user_id` INT NOT NULL,
    `downloaded_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (fps_id) REFERENCES fps_library(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_fps_id (fps_id),
    INDEX idx_user_id (user_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table des commentaires sur les FPS
CREATE TABLE `fps_library_comments` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `fps_id` INT NOT NULL,
    `user_id` INT NOT NULL,
    `parent_id` INT,
    `comment` TEXT NOT NULL,
    `is_approved` BOOLEAN DEFAULT TRUE,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (fps_id) REFERENCES fps_library(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (parent_id) REFERENCES fps_library_comments(id) ON DELETE CASCADE,
    INDEX idx_fps_id (fps_id),
    INDEX idx_user_id (user_id),
    INDEX idx_parent_id (parent_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================================================
-- SYSTÈME DE PARTAGE ET COLLABORATION
-- ===================================================================

-- Table des partages de cycles
CREATE TABLE `cycle_shares` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `cycle_id` INT NOT NULL,
    `shared_by` INT NOT NULL,
    `shared_with` INT NOT NULL,
    `share_type` ENUM('view', 'comment', 'edit', 'duplicate') DEFAULT 'view',
    `message` TEXT,
    `expires_at` TIMESTAMP NULL,
    `accepted_at` TIMESTAMP NULL,
    `status` ENUM('pending', 'accepted', 'rejected', 'expired') DEFAULT 'pending',
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (cycle_id) REFERENCES cycles(id) ON DELETE CASCADE,
    FOREIGN KEY (shared_by) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (shared_with) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_cycle_id (cycle_id),
    INDEX idx_shared_with (shared_with),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table des permissions de partage
CREATE TABLE `share_permissions` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `share_id` INT NOT NULL,
    `permission` ENUM('view', 'comment', 'download', 'duplicate', 'edit') NOT NULL,
    `granted` BOOLEAN DEFAULT TRUE,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (share_id) REFERENCES cycle_shares(id) ON DELETE CASCADE,
    UNIQUE KEY unique_share_permission (share_id, permission)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================================================
-- GAMIFICATION ET PROGRESSION
-- ===================================================================

-- Table des achievements (badges)
CREATE TABLE `achievements` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `code` VARCHAR(50) NOT NULL UNIQUE,
    `name` VARCHAR(100) NOT NULL,
    `description` TEXT,
    `category` VARCHAR(50),
    `icon` VARCHAR(50),
    `color` VARCHAR(7),
    `xp_reward` INT DEFAULT 0,
    `rarity` ENUM('common', 'rare', 'epic', 'legendary') DEFAULT 'common',
    `criteria` JSON,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_category (category),
    INDEX idx_rarity (rarity)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table des achievements débloqués par les utilisateurs
CREATE TABLE `user_achievements` (
    `user_id` INT NOT NULL,
    `achievement_id` INT NOT NULL,
    `unlocked_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `progress` INT DEFAULT 100,
    `metadata` JSON,
    PRIMARY KEY (user_id, achievement_id),
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (achievement_id) REFERENCES achievements(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table de progression d'apprentissage
CREATE TABLE `learning_progress` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT NOT NULL,
    `topic` VARCHAR(100) NOT NULL,
    `module` VARCHAR(100),
    `progress_percentage` INT DEFAULT 0,
    `completed` BOOLEAN DEFAULT FALSE,
    `last_activity` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    `time_spent_minutes` INT DEFAULT 0,
    `metadata` JSON,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_topic (user_id, topic),
    INDEX idx_completed (completed)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================================================
-- SYSTÈME DE NOTIFICATIONS ET LOGS
-- ===================================================================

-- Table des notifications
CREATE TABLE `notifications` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT NOT NULL,
    `type` VARCHAR(100) NOT NULL,
    `title` VARCHAR(255) NOT NULL,
    `message` TEXT,
    `data` JSON,
    `icon` VARCHAR(50),
    `color` VARCHAR(7),
    `action_url` VARCHAR(500),
    `is_read` BOOLEAN DEFAULT FALSE,
    `read_at` TIMESTAMP NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_unread (user_id, is_read),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table des logs d'activité
CREATE TABLE `activity_logs` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT,
    `type` VARCHAR(100) NOT NULL,
    `action` VARCHAR(100) NOT NULL,
    `entity_type` VARCHAR(100),
    `entity_id` INT,
    `message` TEXT,
    `data` JSON,
    `ip_address` VARCHAR(45),
    `user_agent` TEXT,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_user_id (user_id),
    INDEX idx_type_action (type, action),
    INDEX idx_entity (entity_type, entity_id),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================================================
-- SÉCURITÉ
-- ===================================================================

-- Table des tentatives de connexion
CREATE TABLE `login_attempts` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `email` VARCHAR(255),
    `ip_address` VARCHAR(45),
    `success` BOOLEAN DEFAULT FALSE,
    `attempted_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_email_ip (email, ip_address, attempted_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table de réinitialisation de mot de passe
CREATE TABLE `password_resets` (
    `email` VARCHAR(255) NOT NULL,
    `token` VARCHAR(255) NOT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (email),
    INDEX idx_token (token)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ===================================================================
-- INSERTION DES DONNÉES DE BASE
-- ===================================================================

-- Insertion des ECC
INSERT INTO `ecc_definitions` (`code`, `name`, `icon`, `color`, `gradient`, `description`, `imaginaire_symbolique`, `but_activite`, `mobiles_typiques`, `ppsad_examples`) VALUES
('ECC1', 'Performance & Dépassement', 'trophy', '#FF6B6B', 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)', 
 'Explorer ses limites, se dépasser, performer', 
 'Dépassement de soi, conquête héroïque de l\'espace-temps, domination éphémère sur le monde',
 'Produire une performance mesurable dans un cadre normé',
 'Se dépasser, battre son record, explorer ses limites',
 '["Demi-fond", "Natation vitesse", "Relais-vitesse", "Pentabond"]'),

('ECC2', 'Aventure & Exploration', 'compass', '#4ECDC4', 'linear-gradient(135deg, #f093fb 0%, #f5576c 100%)',
 'S\'aventurer, explorer l\'inconnu, être autonome',
 'Explorer l\'inconnu, vivre l\'inattendu, devenir le héros de son épopée',
 'Conduire et adapter son déplacement dans des environnements variés/incertains',
 'Découvrir, s\'aventurer, être autonome dans le milieu',
 '["Escalade", "Course d\'orientation", "Sauvetage", "VTT"]'),

('ECC3', 'Arts du Mouvement', 'theater-masks', '#95E77E', 'linear-gradient(135deg, #fa709a 0%, #fee140 100%)',
 'S\'exprimer, créer, communiquer par le corps',
 'Expression de soi, création, communication sensible',
 'Créer et présenter une composition corporelle porteuse de sens',
 'S\'exprimer, créer, émouvoir, partager',
 '["Danse", "Arts du cirque", "Gymnastique rythmique"]'),

('ECC4', 'Confrontation & Coopération', 'handshake', '#FFE66D', 'linear-gradient(135deg, #a8edea 0%, #fed6e3 100%)',
 'S\'opposer, coopérer, élaborer des stratégies',
 'Affrontement codifié, défi mutuel, reconnaissance dans l\'opposition',
 'S\'opposer individuellement ou collectivement pour gagner',
 'Gagner, coopérer, élaborer des stratégies',
 '["Sports collectifs", "Sports de raquette", "Sports de combat"]'),

('ECC5', 'Découverte & Soin de Soi', 'heart', '#A8DADC', 'linear-gradient(135deg, #ffecd2 0%, #fcb69f 100%)',
 'Se connaître, se transformer, prendre soin de soi',
 'Bien-être, transformation de soi, harmonie corps-esprit',
 'Développer ses ressources pour agir sur soi de façon éclairée',
 'Se connaître, se transformer, prendre soin de soi',
 '["Course en durée", "Musculation", "Step", "Yoga"]'),

('ECC6', 'Acrobatie & Défis', 'star', '#F1FAEE', 'linear-gradient(135deg, #a1c4fd 0%, #c2e9fb 100%)',
 'Défier la gravité, maîtriser le vertige',
 'Jeu avec l\'équilibre et le vertige, défis gravitaires, sensations fortes',
 'Réaliser des actions motrices défiant les lois habituelles',
 'Oser, maîtriser le risque, impressionner',
 '["Gymnastique", "Acrosport", "Trampoline", "Parkour"]');

-- Insertion des rôles système
INSERT INTO `roles` (`code`, `name`, `level`, `icon`, `color`, `description`) VALUES
('SUPER_ADMIN', 'Super Administrateur', 100, 'shield-crown', '#9333EA', 'Accès total au système'),
('INSPECTEUR', 'Inspecteur Académique', 80, 'user-graduate', '#DC2626', 'Supervision pédagogique académique'),
('CHEF_ETABLISSEMENT', 'Chef d\'Établissement', 70, 'building-columns', '#EA580C', 'Direction d\'établissement'),
('COORDINATEUR', 'Coordinateur EPS', 60, 'users-cog', '#0891B2', 'Coordination équipe EPS'),
('FORMATEUR', 'Formateur CEDREPS', 50, 'chalkboard-teacher', '#059669', 'Formation et accompagnement'),
('ENSEIGNANT', 'Enseignant EPS', 30, 'user-tie', '#3B82F6', 'Enseignant d\'EPS'),
('STAGIAIRE', 'Stagiaire', 20, 'user-graduate', '#8B5CF6', 'Enseignant stagiaire'),
('OBSERVATEUR', 'Observateur', 10, 'eye', '#6B7280', 'Accès en lecture seule');

-- Insertion des rôles sociaux pour élèves
INSERT INTO `student_roles` (`name`, `description`, `icon`, `color`) VALUES
('Observateur', 'Observe et analyse la pratique des autres', 'eye', '#3B82F6'),
('Arbitre', 'Assure le respect des règles', 'gavel', '#EF4444'),
('Juge', 'Évalue les performances', 'balance-scale', '#F59E0B'),
('Coach', 'Conseille et encourage', 'bullhorn', '#10B981'),
('Chronométreur', 'Gère le temps', 'stopwatch', '#8B5CF6'),
('Secrétaire', 'Note les résultats', 'clipboard', '#EC4899'),
('Pareur', 'Assure la sécurité', 'hands-helping', '#14B8A6'),
('Capitaine', 'Dirige l\'équipe', 'crown', '#F97316');

-- Insertion des achievements de base
INSERT INTO `achievements` (`code`, `name`, `description`, `category`, `icon`, `color`, `xp_reward`, `rarity`) VALUES
('FIRST_CYCLE', 'Premier Cycle', 'Créer votre premier cycle', 'creation', 'flag', '#10B981', 100, 'common'),
('FIVE_FPS', '5 FPS Créées', 'Créer 5 Formes de Pratique Scolaire', 'creation', 'cogs', '#3B82F6', 250, 'rare'),
('ALL_ECC', 'Explorateur ECC', 'Créer au moins un cycle pour chaque ECC', 'exploration', 'compass', '#8B5CF6', 500, 'epic'),
('PERFECT_CONFORMITY', 'Conformité Parfaite', 'Atteindre 100% de conformité CEDREPS', 'excellence', 'award', '#F59E0B', 1000, 'legendary'),
('COLLABORATOR', 'Collaborateur', 'Partager 10 cycles avec des collègues', 'social', 'share-alt', '#06B6D4', 300, 'rare'),
('EVALUATOR', 'Évaluateur Expert', 'Évaluer 100 élèves', 'evaluation', 'chart-line', '#EF4444', 400, 'rare');

COMMIT;