#!/usr/bin/env python3
"""
Diagnostic complet EPSWeb - Test de l'erreur HTTP 500
Ce script teste toutes les pages et fonctionnalités du site epsweb.fr
"""

import requests
import json
import time
from datetime import datetime
from colorama import init, Fore, Style
import sys
import traceback
from urllib.parse import urljoin

# Initialisation de colorama pour les couleurs dans le terminal
init(autoreset=True)

class EPSWebDiagnostic:
    def __init__(self):
        self.base_url = "https://epsweb.fr"
        self.app_url = "https://epsweb.fr/app_v5/"
        self.session = requests.Session()
        self.results = []
        
        # Comptes de test
        self.accounts = {
            "demo1": {
                "email": "demo@cedreps.fr",
                "password": "Demo2025!",
                "name": "Demo CEDREPS"
            },
            "demo2": {
                "email": "demo@epsweb.fr",
                "password": "Test123!",
                "name": "Demo EPSWeb"
            }
        }
        
        # Pages à tester
        self.pages_to_test = [
            {"url": "/", "name": "Page d'accueil", "requires_auth": False},
            {"url": "/app_v5/", "name": "Application v5", "requires_auth": False},
            {"url": "/app_v5/login.php", "name": "Page de connexion", "requires_auth": False},
            {"url": "/app_v5/dashboard.php", "name": "Dashboard", "requires_auth": True},
            {"url": "/app_v5/index.php", "name": "Index App", "requires_auth": False},
            {"url": "/app_v5/profile.php", "name": "Profil", "requires_auth": True},
            {"url": "/app_v5/settings.php", "name": "Paramètres", "requires_auth": True},
        ]
        
    def print_header(self):
        """Affiche l'en-tête du diagnostic"""
        print(f"\n{Fore.CYAN}{'='*60}")
        print(f"{Fore.CYAN}🔍 DIAGNOSTIC EPSWEB - ERREUR HTTP 500")
        print(f"{Fore.CYAN}{'='*60}")
        print(f"{Fore.WHITE}Date : {datetime.now().strftime('%d/%m/%Y %H:%M:%S')}")
        print(f"{Fore.WHITE}URL cible : {self.base_url}")
        print(f"{Fore.CYAN}{'='*60}\n")
        
    def test_basic_connectivity(self):
        """Test de connectivité basique au serveur"""
        print(f"{Fore.YELLOW}[1] Test de connectivité basique...")
        
        try:
            response = self.session.get(self.base_url, timeout=10, verify=True)
            
            if response.status_code == 200:
                print(f"{Fore.GREEN}✓ Serveur accessible (Code HTTP: {response.status_code})")
                print(f"{Fore.WHITE}  - Temps de réponse: {response.elapsed.total_seconds():.2f}s")
                print(f"{Fore.WHITE}  - Taille de la réponse: {len(response.content)} octets")
                
                # Vérification des headers
                print(f"\n{Fore.CYAN}Headers du serveur:")
                important_headers = ['Server', 'X-Powered-By', 'Content-Type', 'Set-Cookie']
                for header in important_headers:
                    if header in response.headers:
                        print(f"  {header}: {response.headers[header]}")
                        
                return True
                
            else:
                print(f"{Fore.RED}✗ Erreur HTTP {response.status_code}")
                return False
                
        except requests.exceptions.RequestException as e:
            print(f"{Fore.RED}✗ Erreur de connexion: {e}")
            return False
            
    def test_login(self, account_key="demo1"):
        """Test de connexion avec un compte"""
        account = self.accounts[account_key]
        print(f"\n{Fore.YELLOW}[2] Test de connexion avec {account['name']}...")
        print(f"{Fore.WHITE}  Email: {account['email']}")
        
        login_url = urljoin(self.app_url, "login.php")
        
        try:
            # GET sur la page de login pour récupérer les cookies/tokens CSRF
            print(f"{Fore.WHITE}  - Récupération de la page de connexion...")
            response = self.session.get(login_url, timeout=10)
            
            if response.status_code != 200:
                print(f"{Fore.RED}  ✗ Page de connexion inaccessible (HTTP {response.status_code})")
                return False
                
            # Tentative de POST pour se connecter
            print(f"{Fore.WHITE}  - Envoi des identifiants...")
            login_data = {
                "email": account["email"],
                "password": account["password"],
                "submit": "Se connecter"
            }
            
            response = self.session.post(
                login_url,
                data=login_data,
                allow_redirects=True,
                timeout=10
            )
            
            # Analyse de la réponse
            if response.status_code == 200:
                if "dashboard" in response.url.lower() or "Déconnexion" in response.text:
                    print(f"{Fore.GREEN}  ✓ Connexion réussie!")
                    print(f"{Fore.WHITE}    - Redirection vers: {response.url}")
                    return True
                else:
                    print(f"{Fore.YELLOW}  ⚠ Page chargée mais connexion incertaine")
                    print(f"{Fore.WHITE}    - URL actuelle: {response.url}")
                    return False
                    
            elif response.status_code == 500:
                print(f"{Fore.RED}  ✗ ERREUR HTTP 500 DÉTECTÉE!")
                self.analyze_error_500(response)
                return False
                
            else:
                print(f"{Fore.RED}  ✗ Erreur HTTP {response.status_code}")
                return False
                
        except Exception as e:
            print(f"{Fore.RED}  ✗ Erreur lors de la connexion: {e}")
            return False
            
    def analyze_error_500(self, response):
        """Analyse détaillée d'une erreur 500"""
        print(f"\n{Fore.RED}{'='*50}")
        print(f"{Fore.RED}ANALYSE DE L'ERREUR HTTP 500")
        print(f"{Fore.RED}{'='*50}")
        
        print(f"\n{Fore.YELLOW}Informations de la réponse:")
        print(f"  - URL: {response.url}")
        print(f"  - Code: {response.status_code}")
        print(f"  - Headers: {dict(response.headers)}")
        
        # Recherche d'indices dans le contenu
        if response.content:
            content = response.text[:1000]  # Premiers 1000 caractères
            
            # Recherche de messages d'erreur PHP
            error_patterns = [
                "Fatal error",
                "Parse error",
                "Warning:",
                "Notice:",
                "Exception",
                "Stack trace",
                "mysqli_",
                "mysql_",
                "PDO",
                "Database"
            ]
            
            print(f"\n{Fore.YELLOW}Recherche d'indices dans la réponse:")
            for pattern in error_patterns:
                if pattern.lower() in content.lower():
                    print(f"{Fore.RED}  ⚠ Trouvé: {pattern}")
                    
            # Affichage d'un extrait du contenu si disponible
            if len(content) > 0:
                print(f"\n{Fore.YELLOW}Extrait du contenu (premiers 500 caractères):")
                print(f"{Fore.WHITE}{content[:500]}")
                
    def test_all_pages(self, authenticated=False):
        """Test de toutes les pages"""
        print(f"\n{Fore.YELLOW}[3] Test de toutes les pages...")
        
        success_count = 0
        error_500_count = 0
        other_errors = 0
        
        for page in self.pages_to_test:
            if page["requires_auth"] and not authenticated:
                continue
                
            url = urljoin(self.base_url, page["url"])
            print(f"\n{Fore.WHITE}Testing: {page['name']} ({url})")
            
            try:
                start_time = time.time()
                response = self.session.get(url, timeout=10, allow_redirects=True)
                elapsed_time = (time.time() - start_time) * 1000  # en ms
                
                if response.status_code == 200:
                    print(f"{Fore.GREEN}  ✓ OK (200) - {elapsed_time:.0f}ms")
                    success_count += 1
                    
                elif response.status_code == 500:
                    print(f"{Fore.RED}  ✗ ERREUR 500 DÉTECTÉE!")
                    error_500_count += 1
                    self.analyze_error_500(response)
                    
                elif response.status_code in [301, 302, 303, 307, 308]:
                    print(f"{Fore.YELLOW}  → Redirection ({response.status_code}) vers {response.headers.get('Location', 'Unknown')}")
                    
                elif response.status_code == 403:
                    print(f"{Fore.YELLOW}  ⚠ Accès refusé (403)")
                    
                elif response.status_code == 404:
                    print(f"{Fore.YELLOW}  ⚠ Page non trouvée (404)")
                    other_errors += 1
                    
                else:
                    print(f"{Fore.YELLOW}  ⚠ Code HTTP: {response.status_code}")
                    other_errors += 1
                    
                self.results.append({
                    "page": page["name"],
                    "url": url,
                    "status_code": response.status_code,
                    "response_time": elapsed_time,
                    "timestamp": datetime.now().isoformat()
                })
                
            except requests.exceptions.Timeout:
                print(f"{Fore.RED}  ✗ Timeout")
                other_errors += 1
                
            except Exception as e:
                print(f"{Fore.RED}  ✗ Erreur: {e}")
                other_errors += 1
                
        # Résumé
        print(f"\n{Fore.CYAN}{'='*50}")
        print(f"{Fore.CYAN}RÉSUMÉ DES TESTS")
        print(f"{Fore.CYAN}{'='*50}")
        print(f"{Fore.GREEN}Succès: {success_count}")
        print(f"{Fore.RED}Erreurs 500: {error_500_count}")
        print(f"{Fore.YELLOW}Autres erreurs: {other_errors}")
        
        return error_500_count
        
    def test_database_endpoints(self):
        """Test des endpoints liés à la base de données"""
        print(f"\n{Fore.YELLOW}[4] Test des endpoints de base de données...")
        
        db_endpoints = [
            "/app_v5/api/check_db.php",
            "/app_v5/api/users.php",
            "/app_v5/includes/db_connect.php",
            "/app_v5/config.php"
        ]
        
        for endpoint in db_endpoints:
            url = urljoin(self.base_url, endpoint)
            print(f"\n{Fore.WHITE}Testing: {url}")
            
            try:
                response = self.session.get(url, timeout=5)
                if response.status_code == 500:
                    print(f"{Fore.RED}  ✗ ERREUR 500 - Possible problème de base de données")
                elif response.status_code == 200:
                    print(f"{Fore.GREEN}  ✓ Accessible (200)")
                elif response.status_code == 404:
                    print(f"{Fore.YELLOW}  - Non trouvé (404)")
                else:
                    print(f"{Fore.YELLOW}  - Code: {response.status_code}")
            except Exception as e:
                print(f"{Fore.WHITE}  - Erreur: {e}")
                
    def generate_report(self):
        """Génère un rapport détaillé"""
        print(f"\n{Fore.CYAN}{'='*50}")
        print(f"{Fore.CYAN}RAPPORT DE DIAGNOSTIC")
        print(f"{Fore.CYAN}{'='*50}")
        
        print(f"\n{Fore.YELLOW}Recommandations pour résoudre l'erreur HTTP 500:")
        
        recommendations = [
            "1. Vérifier les logs PHP (/var/log/apache2/error.log ou error_log)",
            "2. Activer temporairement display_errors dans php.ini:",
            "   display_errors = On",
            "   error_reporting = E_ALL",
            "3. Vérifier la connexion à la base de données:",
            "   - Host, utilisateur, mot de passe",
            "   - Privilèges de l'utilisateur MySQL",
            "4. Vérifier les permissions des fichiers:",
            "   - chmod 755 pour les dossiers",
            "   - chmod 644 pour les fichiers PHP",
            "5. Vérifier la syntaxe PHP:",
            "   php -l dashboard.php",
            "6. Vérifier les dépendances/includes manquants",
            "7. Tester avec un fichier PHP simple pour isoler le problème"
        ]
        
        for rec in recommendations:
            print(f"{Fore.WHITE}{rec}")
            
        # Sauvegarde des résultats
        filename = f"diagnostic_epsweb_{datetime.now().strftime('%Y%m%d_%H%M%S')}.json"
        with open(filename, 'w', encoding='utf-8') as f:
            json.dump(self.results, f, indent=2, ensure_ascii=False)
            
        print(f"\n{Fore.GREEN}✓ Rapport sauvegardé dans: {filename}")
        
    def run_diagnostic(self):
        """Lance le diagnostic complet"""
        self.print_header()
        
        # Test 1: Connectivité
        if not self.test_basic_connectivity():
            print(f"\n{Fore.RED}⚠ Impossible de continuer - serveur inaccessible")
            return
            
        # Test 2: Connexion
        authenticated = self.test_login("demo1")
        if not authenticated:
            print(f"\n{Fore.YELLOW}Tentative avec le second compte...")
            authenticated = self.test_login("demo2")
            
        # Test 3: Pages
        error_count = self.test_all_pages(authenticated)
        
        # Test 4: Endpoints DB
        self.test_database_endpoints()
        
        # Génération du rapport
        self.generate_report()
        
        print(f"\n{Fore.CYAN}{'='*60}")
        print(f"{Fore.GREEN}DIAGNOSTIC TERMINÉ")
        
        if error_count > 0:
            print(f"{Fore.RED}⚠ {error_count} erreur(s) HTTP 500 détectée(s)")
            print(f"{Fore.YELLOW}Consultez les logs du serveur pour plus de détails")
        else:
            print(f"{Fore.GREEN}✓ Aucune erreur 500 détectée dans les tests")
            
        print(f"{Fore.CYAN}{'='*60}\n")

if __name__ == "__main__":
    print(f"{Fore.CYAN}Installation des dépendances si nécessaire...")
    print(f"{Fore.WHITE}pip install requests colorama")
    
    try:
        diagnostic = EPSWebDiagnostic()
        diagnostic.run_diagnostic()
    except KeyboardInterrupt:
        print(f"\n{Fore.YELLOW}Diagnostic interrompu par l'utilisateur")
        sys.exit(0)
    except Exception as e:
        print(f"\n{Fore.RED}Erreur fatale: {e}")
        traceback.print_exc()
        sys.exit(1)