#!/bin/bash

# Script de diagnostic EPSWeb avec curl
# Utilisation: ./diagnostic_epsweb.sh

# Couleurs pour l'affichage
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Configuration
BASE_URL="https://epsweb.fr"
APP_URL="${BASE_URL}/app_v5"
COOKIE_FILE="/tmp/epsweb_cookies.txt"

# Comptes de test
DEMO1_EMAIL="demo@cedreps.fr"
DEMO1_PASSWORD="Demo2025!"

DEMO2_EMAIL="demo@epsweb.fr"
DEMO2_PASSWORD="Test123!"

# Fonction pour afficher l'en-tête
print_header() {
    echo -e "${BLUE}========================================================${NC}"
    echo -e "${BLUE}     DIAGNOSTIC EPSWEB - ERREUR HTTP 500${NC}"
    echo -e "${BLUE}========================================================${NC}"
    echo -e "Date: $(date '+%d/%m/%Y %H:%M:%S')"
    echo -e "URL cible: ${BASE_URL}"
    echo -e "${BLUE}========================================================${NC}\n"
}

# Fonction pour tester une URL
test_url() {
    local url=$1
    local name=$2
    local auth_required=$3
    
    echo -e "\n${YELLOW}Testing: ${name}${NC}"
    echo -e "URL: ${url}"
    
    # Effectuer la requête avec curl
    response=$(curl -s -o /dev/null -w "%{http_code}|%{time_total}|%{size_download}" \
               -b "${COOKIE_FILE}" \
               -L \
               --max-time 10 \
               "${url}")
    
    # Parser la réponse
    IFS='|' read -r http_code time_total size_download <<< "$response"
    
    # Convertir le temps en millisecondes
    time_ms=$(echo "$time_total * 1000" | bc | cut -d. -f1)
    
    # Afficher le résultat avec couleur
    case $http_code in
        200)
            echo -e "${GREEN}✓ OK (${http_code}) - ${time_ms}ms - ${size_download} bytes${NC}"
            ;;
        500)
            echo -e "${RED}✗ ERREUR 500 DÉTECTÉE!${NC}"
            echo -e "${RED}Cette page génère l'erreur HTTP 500${NC}"
            
            # Essayer de récupérer plus de détails
            echo -e "\n${YELLOW}Tentative de récupération du contenu d'erreur:${NC}"
            curl -s -b "${COOKIE_FILE}" -L --max-time 10 "${url}" | head -n 20
            ;;
        301|302|303|307|308)
            echo -e "${YELLOW}→ Redirection (${http_code})${NC}"
            ;;
        401|403)
            echo -e "${YELLOW}⚠ Accès refusé (${http_code})${NC}"
            ;;
        404)
            echo -e "${YELLOW}⚠ Page non trouvée (${http_code})${NC}"
            ;;
        000)
            echo -e "${RED}✗ Timeout ou erreur de connexion${NC}"
            ;;
        *)
            echo -e "${YELLOW}⚠ Code HTTP: ${http_code}${NC}"
            ;;
    esac
}

# Fonction pour tester la connexion
test_login() {
    local email=$1
    local password=$2
    local account_name=$3
    
    echo -e "\n${YELLOW}[Test de connexion avec ${account_name}]${NC}"
    echo -e "Email: ${email}"
    
    # Nettoyer les cookies précédents
    rm -f "${COOKIE_FILE}"
    
    # Récupérer la page de login (pour les cookies de session)
    echo -e "1. Récupération de la page de connexion..."
    curl -s -c "${COOKIE_FILE}" -o /dev/null "${APP_URL}/login.php"
    
    # Tentative de connexion
    echo -e "2. Envoi des identifiants..."
    response=$(curl -s -w "\n%{http_code}" \
               -b "${COOKIE_FILE}" \
               -c "${COOKIE_FILE}" \
               -L \
               -d "email=${email}" \
               -d "password=${password}" \
               -d "submit=Se connecter" \
               "${APP_URL}/login.php")
    
    http_code=$(echo "$response" | tail -n 1)
    content=$(echo "$response" | head -n -1)
    
    # Analyser la réponse
    if [ "$http_code" = "200" ]; then
        if echo "$content" | grep -q "dashboard\|Déconnexion\|Logout"; then
            echo -e "${GREEN}✓ Connexion réussie!${NC}"
            return 0
        else
            echo -e "${YELLOW}⚠ Page chargée mais connexion incertaine${NC}"
            return 1
        fi
    elif [ "$http_code" = "500" ]; then
        echo -e "${RED}✗ ERREUR HTTP 500 lors de la connexion!${NC}"
        echo -e "${RED}Le problème se produit pendant le processus de connexion${NC}"
        return 1
    else
        echo -e "${RED}✗ Erreur HTTP ${http_code}${NC}"
        return 1
    fi
}

# Fonction pour analyser les headers
analyze_headers() {
    local url=$1
    
    echo -e "\n${YELLOW}[Analyse des headers HTTP]${NC}"
    echo -e "URL: ${url}"
    
    headers=$(curl -s -I -L --max-time 10 "${url}")
    
    echo -e "\n${BLUE}Headers reçus:${NC}"
    echo "$headers" | grep -E "HTTP/|Server:|X-Powered-By:|Content-Type:|Set-Cookie:"
}

# Fonction pour tester avec différentes méthodes HTTP
test_http_methods() {
    local url=$1
    
    echo -e "\n${YELLOW}[Test des méthodes HTTP]${NC}"
    echo -e "URL: ${url}"
    
    methods=("GET" "POST" "HEAD" "OPTIONS")
    
    for method in "${methods[@]}"; do
        response=$(curl -s -o /dev/null -w "%{http_code}" -X "$method" --max-time 5 "$url")
        
        case $response in
            200|204|301|302)
                echo -e "${GREEN}$method: $response${NC}"
                ;;
            500)
                echo -e "${RED}$method: 500 - ERREUR SERVEUR!${NC}"
                ;;
            *)
                echo -e "${YELLOW}$method: $response${NC}"
                ;;
        esac
    done
}

# Fonction pour générer un rapport
generate_report() {
    local filename="diagnostic_epsweb_$(date +%Y%m%d_%H%M%S).txt"
    
    echo -e "\n${YELLOW}[Génération du rapport]${NC}"
    
    {
        echo "RAPPORT DE DIAGNOSTIC EPSWEB"
        echo "============================="
        echo "Date: $(date)"
        echo "URL: ${BASE_URL}"
        echo ""
        echo "RÉSUMÉ DES TESTS"
        echo "----------------"
        echo ""
        echo "Pages testées:"
        test_url "${BASE_URL}/" "Page d'accueil" false
        test_url "${APP_URL}/" "Application v5" false
        test_url "${APP_URL}/login.php" "Page de connexion" false
        test_url "${APP_URL}/dashboard.php" "Dashboard" true
        echo ""
        echo "RECOMMANDATIONS"
        echo "---------------"
        echo "1. Vérifier les logs du serveur:"
        echo "   sudo tail -f /var/log/apache2/error.log"
        echo ""
        echo "2. Activer le mode debug PHP:"
        echo "   Ajouter dans le fichier PHP:"
        echo "   ini_set('display_errors', 1);"
        echo "   error_reporting(E_ALL);"
        echo ""
        echo "3. Vérifier la base de données:"
        echo "   - Tester la connexion MySQL"
        echo "   - Vérifier les privilèges utilisateur"
        echo ""
        echo "4. Vérifier les permissions:"
        echo "   chmod 755 pour les dossiers"
        echo "   chmod 644 pour les fichiers PHP"
    } > "$filename"
    
    echo -e "${GREEN}✓ Rapport sauvegardé dans: ${filename}${NC}"
}

# Menu principal
main_menu() {
    while true; do
        echo -e "\n${BLUE}========== MENU PRINCIPAL ==========${NC}"
        echo "1. Test de connectivité basique"
        echo "2. Test de connexion (compte demo1)"
        echo "3. Test de connexion (compte demo2)"
        echo "4. Test de toutes les pages"
        echo "5. Analyse des headers HTTP"
        echo "6. Test dashboard.php (erreur 500)"
        echo "7. Test des méthodes HTTP"
        echo "8. Diagnostic complet"
        echo "9. Générer un rapport"
        echo "0. Quitter"
        echo -e "${BLUE}====================================${NC}"
        
        read -p "Votre choix: " choice
        
        case $choice in
            1)
                test_url "${BASE_URL}" "Page d'accueil" false
                ;;
            2)
                test_login "${DEMO1_EMAIL}" "${DEMO1_PASSWORD}" "Demo CEDREPS"
                ;;
            3)
                test_login "${DEMO2_EMAIL}" "${DEMO2_PASSWORD}" "Demo EPSWeb"
                ;;
            4)
                echo -e "\n${YELLOW}[Test de toutes les pages]${NC}"
                test_url "${BASE_URL}/" "Page d'accueil" false
                test_url "${APP_URL}/" "Application v5" false
                test_url "${APP_URL}/login.php" "Page de connexion" false
                test_url "${APP_URL}/dashboard.php" "Dashboard" true
                test_url "${APP_URL}/index.php" "Index App" false
                ;;
            5)
                analyze_headers "${APP_URL}/dashboard.php"
                ;;
            6)
                echo -e "\n${YELLOW}[Test spécifique de dashboard.php]${NC}"
                test_url "${APP_URL}/dashboard.php" "Dashboard (Erreur 500)" true
                test_http_methods "${APP_URL}/dashboard.php"
                ;;
            7)
                test_http_methods "${APP_URL}/dashboard.php"
                ;;
            8)
                echo -e "\n${YELLOW}[Diagnostic complet]${NC}"
                test_url "${BASE_URL}" "Page d'accueil" false
                test_login "${DEMO1_EMAIL}" "${DEMO1_PASSWORD}" "Demo CEDREPS"
                test_url "${APP_URL}/dashboard.php" "Dashboard" true
                analyze_headers "${APP_URL}/dashboard.php"
                generate_report
                ;;
            9)
                generate_report
                ;;
            0)
                echo -e "\n${GREEN}Au revoir!${NC}"
                rm -f "${COOKIE_FILE}"
                exit 0
                ;;
            *)
                echo -e "${RED}Choix invalide${NC}"
                ;;
        esac
    done
}

# Programme principal
print_header

# Vérifier que curl est installé
if ! command -v curl &> /dev/null; then
    echo -e "${RED}Erreur: curl n'est pas installé${NC}"
    echo "Installez curl avec: sudo apt-get install curl"
    exit 1
fi

# Vérifier que bc est installé (pour les calculs)
if ! command -v bc &> /dev/null; then
    echo -e "${YELLOW}Avertissement: bc n'est pas installé (temps en ms non disponible)${NC}"
    echo "Installez bc avec: sudo apt-get install bc"
fi

# Lancer le menu
main_menu