<?php
/**
 * Script de diagnostic EPSWeb - À uploader sur le serveur
 * Ce script aide à identifier la cause de l'erreur HTTP 500
 * 
 * IMPORTANT: Supprimer ce fichier après utilisation pour des raisons de sécurité!
 */

// Activer l'affichage des erreurs pour le diagnostic
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

// Définir le timezone
date_default_timezone_set('Europe/Paris');

// Démarrer la session pour les tests
@session_start();

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Diagnostic EPSWeb - Serveur</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1000px;
            margin: 0 auto;
            background: white;
            border-radius: 10px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        }
        
        h1 {
            color: #333;
            margin-bottom: 10px;
            border-bottom: 3px solid #667eea;
            padding-bottom: 10px;
        }
        
        .alert {
            padding: 15px;
            margin: 20px 0;
            border-radius: 5px;
            border-left: 4px solid;
        }
        
        .alert-danger {
            background: #fee;
            border-color: #f56565;
            color: #c53030;
        }
        
        .alert-warning {
            background: #fffaf0;
            border-color: #ed8936;
            color: #c05621;
        }
        
        .alert-success {
            background: #f0fdf4;
            border-color: #48bb78;
            color: #22543d;
        }
        
        .alert-info {
            background: #ebf8ff;
            border-color: #4299e1;
            color: #2c5282;
        }
        
        .test-section {
            margin: 30px 0;
            padding: 20px;
            background: #f8f9fa;
            border-radius: 8px;
        }
        
        .test-section h2 {
            color: #667eea;
            margin-bottom: 15px;
            font-size: 20px;
        }
        
        .test-result {
            margin: 10px 0;
            padding: 10px;
            background: white;
            border-radius: 5px;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }
        
        .test-label {
            font-weight: 500;
            color: #333;
        }
        
        .test-value {
            font-family: 'Courier New', monospace;
            background: #f1f5f9;
            padding: 5px 10px;
            border-radius: 3px;
        }
        
        .status-ok {
            color: #48bb78;
            font-weight: bold;
        }
        
        .status-error {
            color: #f56565;
            font-weight: bold;
        }
        
        .status-warning {
            color: #ed8936;
            font-weight: bold;
        }
        
        pre {
            background: #1a202c;
            color: #e2e8f0;
            padding: 15px;
            border-radius: 5px;
            overflow-x: auto;
            margin: 10px 0;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 15px 0;
        }
        
        th, td {
            padding: 10px;
            text-align: left;
            border-bottom: 1px solid #e2e8f0;
        }
        
        th {
            background: #f7fafc;
            font-weight: 600;
            color: #2d3748;
        }
        
        .button {
            background: #667eea;
            color: white;
            padding: 10px 20px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-weight: 500;
            margin: 5px;
        }
        
        .button:hover {
            background: #5a67d8;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔍 Diagnostic EPSWeb - Analyse Serveur</h1>
        
        <div class="alert alert-danger">
            <strong>⚠️ Sécurité:</strong> Ce fichier expose des informations sensibles. 
            Supprimez-le immédiatement après utilisation!
        </div>

        <?php
        // Fonction pour afficher un résultat de test
        function displayTest($label, $value, $status = 'ok') {
            $statusClass = 'status-' . $status;
            $icon = $status == 'ok' ? '✅' : ($status == 'error' ? '❌' : '⚠️');
            echo "<div class='test-result'>";
            echo "<span class='test-label'>$label</span>";
            echo "<span class='test-value $statusClass'>$icon $value</span>";
            echo "</div>";
        }
        ?>

        <!-- Test 1: Configuration PHP -->
        <div class="test-section">
            <h2>1. Configuration PHP</h2>
            <?php
            displayTest('Version PHP', PHP_VERSION, version_compare(PHP_VERSION, '7.0.0', '>=') ? 'ok' : 'warning');
            displayTest('Memory Limit', ini_get('memory_limit'));
            displayTest('Max Execution Time', ini_get('max_execution_time') . ' secondes');
            displayTest('Display Errors', ini_get('display_errors') ? 'Activé' : 'Désactivé', ini_get('display_errors') ? 'warning' : 'ok');
            displayTest('Error Reporting', error_reporting());
            displayTest('Post Max Size', ini_get('post_max_size'));
            displayTest('Upload Max Filesize', ini_get('upload_max_filesize'));
            ?>
        </div>

        <!-- Test 2: Extensions PHP -->
        <div class="test-section">
            <h2>2. Extensions PHP Requises</h2>
            <?php
            $required_extensions = ['mysqli', 'pdo', 'pdo_mysql', 'json', 'session', 'mbstring', 'gd', 'curl'];
            foreach ($required_extensions as $ext) {
                $loaded = extension_loaded($ext);
                displayTest("Extension $ext", $loaded ? 'Installée' : 'Manquante', $loaded ? 'ok' : 'error');
            }
            ?>
        </div>

        <!-- Test 3: Connexion Base de Données -->
        <div class="test-section">
            <h2>3. Test de Connexion à la Base de Données</h2>
            <?php
            // Recherche du fichier de configuration
            $config_files = [
                'config.php',
                'includes/config.php',
                'includes/db_config.php',
                '../config.php',
                '../includes/config.php'
            ];
            
            $config_found = false;
            foreach ($config_files as $file) {
                if (file_exists($file)) {
                    displayTest("Fichier de configuration", $file, 'ok');
                    $config_found = true;
                    
                    // Tentative de connexion
                    try {
                        // Essayer d'inclure le fichier de config
                        @include_once($file);
                        
                        // Test avec mysqli si les constantes existent
                        if (defined('DB_HOST') && defined('DB_USER') && defined('DB_PASS') && defined('DB_NAME')) {
                            $connection = @mysqli_connect(DB_HOST, DB_USER, DB_PASS, DB_NAME);
                            
                            if ($connection) {
                                displayTest("Connexion MySQL", "Réussie", 'ok');
                                displayTest("Serveur MySQL", mysqli_get_server_info($connection), 'ok');
                                
                                // Test d'une requête simple
                                $result = mysqli_query($connection, "SELECT COUNT(*) as count FROM information_schema.tables WHERE table_schema = '" . DB_NAME . "'");
                                if ($result) {
                                    $row = mysqli_fetch_assoc($result);
                                    displayTest("Nombre de tables", $row['count'], 'ok');
                                }
                                
                                mysqli_close($connection);
                            } else {
                                displayTest("Connexion MySQL", "Échec: " . mysqli_connect_error(), 'error');
                            }
                        } else {
                            displayTest("Configuration DB", "Constantes non définies", 'warning');
                        }
                    } catch (Exception $e) {
                        displayTest("Erreur", $e->getMessage(), 'error');
                    }
                    break;
                }
            }
            
            if (!$config_found) {
                displayTest("Fichier de configuration", "Non trouvé", 'error');
            }
            ?>
        </div>

        <!-- Test 4: Permissions des fichiers -->
        <div class="test-section">
            <h2>4. Permissions des Fichiers et Dossiers</h2>
            <?php
            $paths_to_check = [
                '.' => 'Dossier actuel',
                '../uploads' => 'Dossier uploads',
                '../cache' => 'Dossier cache',
                '../logs' => 'Dossier logs',
                'dashboard.php' => 'Fichier dashboard.php'
            ];
            
            foreach ($paths_to_check as $path => $label) {
                if (file_exists($path)) {
                    $perms = substr(sprintf('%o', fileperms($path)), -4);
                    $writable = is_writable($path) ? 'Écriture OK' : 'Lecture seule';
                    displayTest($label, "$perms - $writable", is_writable($path) ? 'ok' : 'warning');
                } else {
                    displayTest($label, "N'existe pas", 'warning');
                }
            }
            ?>
        </div>

        <!-- Test 5: Session -->
        <div class="test-section">
            <h2>5. Test de Session</h2>
            <?php
            displayTest("Session ID", session_id() ?: 'Non démarrée', session_id() ? 'ok' : 'error');
            displayTest("Session Save Path", session_save_path() ?: 'Par défaut');
            
            // Test d'écriture de session
            $_SESSION['test'] = 'test_value_' . time();
            displayTest("Écriture de session", isset($_SESSION['test']) ? 'OK' : 'Échec', isset($_SESSION['test']) ? 'ok' : 'error');
            ?>
        </div>

        <!-- Test 6: Analyse du fichier dashboard.php -->
        <div class="test-section">
            <h2>6. Analyse du fichier dashboard.php</h2>
            <?php
            $dashboard_file = 'dashboard.php';
            if (file_exists($dashboard_file)) {
                // Vérification de la syntaxe PHP
                $output = shell_exec("php -l $dashboard_file 2>&1");
                if (strpos($output, 'No syntax errors') !== false) {
                    displayTest("Syntaxe PHP", "Valide", 'ok');
                } else {
                    displayTest("Syntaxe PHP", "Erreurs détectées", 'error');
                    echo "<pre>$output</pre>";
                }
                
                // Analyse du contenu
                $content = file_get_contents($dashboard_file);
                
                // Recherche de problèmes courants
                $issues = [];
                if (strpos($content, 'include') !== false || strpos($content, 'require') !== false) {
                    $issues[] = "Utilise des includes/requires - vérifier les chemins";
                }
                if (strpos($content, 'mysqli_') !== false || strpos($content, 'mysql_') !== false) {
                    $issues[] = "Utilise des fonctions MySQL - vérifier la connexion DB";
                }
                if (strpos($content, 'session_start') === false) {
                    $issues[] = "Pas de session_start() détecté";
                }
                
                if (!empty($issues)) {
                    echo "<div class='alert alert-warning'>";
                    echo "<strong>Points d'attention:</strong><ul>";
                    foreach ($issues as $issue) {
                        echo "<li>$issue</li>";
                    }
                    echo "</ul></div>";
                }
                
                displayTest("Taille du fichier", filesize($dashboard_file) . ' octets', 'ok');
                displayTest("Dernière modification", date('d/m/Y H:i:s', filemtime($dashboard_file)), 'ok');
            } else {
                displayTest("Fichier dashboard.php", "Non trouvé", 'error');
            }
            ?>
        </div>

        <!-- Test 7: Logs d'erreur -->
        <div class="test-section">
            <h2>7. Dernières Erreurs PHP</h2>
            <?php
            $error_log = ini_get('error_log');
            if ($error_log && file_exists($error_log)) {
                displayTest("Fichier de log", $error_log, 'ok');
                
                // Lire les dernières lignes
                $lines = file($error_log);
                $last_errors = array_slice($lines, -10);
                
                if (!empty($last_errors)) {
                    echo "<div class='alert alert-info'>Dernières erreurs:</div>";
                    echo "<pre>";
                    foreach ($last_errors as $error) {
                        echo htmlspecialchars($error);
                    }
                    echo "</pre>";
                }
            } else {
                displayTest("Fichier de log", "Non accessible", 'warning');
            }
            ?>
        </div>

        <!-- Test 8: Variables d'environnement -->
        <div class="test-section">
            <h2>8. Environnement Serveur</h2>
            <table>
                <tr>
                    <th>Variable</th>
                    <th>Valeur</th>
                </tr>
                <?php
                $server_vars = [
                    'SERVER_SOFTWARE',
                    'SERVER_NAME',
                    'DOCUMENT_ROOT',
                    'SCRIPT_FILENAME',
                    'REQUEST_URI',
                    'PHP_SELF'
                ];
                
                foreach ($server_vars as $var) {
                    echo "<tr>";
                    echo "<td><strong>$var</strong></td>";
                    echo "<td>" . ($_SERVER[$var] ?? 'Non défini') . "</td>";
                    echo "</tr>";
                }
                ?>
            </table>
        </div>

        <!-- Recommandations -->
        <div class="test-section">
            <h2>📋 Recommandations pour résoudre l'erreur 500</h2>
            <ol>
                <li><strong>Activer le mode debug:</strong>
                    <pre>ini_set('display_errors', 1);
error_reporting(E_ALL);</pre>
                </li>
                <li><strong>Vérifier les logs Apache/Nginx:</strong>
                    <pre>tail -f /var/log/apache2/error.log
# ou
tail -f /var/log/nginx/error.log</pre>
                </li>
                <li><strong>Tester la connexion DB manuellement:</strong>
                    <pre>mysql -u [username] -p[password] -h [host] [database]</pre>
                </li>
                <li><strong>Corriger les permissions:</strong>
                    <pre>find . -type f -exec chmod 644 {} \;
find . -type d -exec chmod 755 {} \;</pre>
                </li>
                <li><strong>Vider le cache si présent:</strong>
                    <pre>rm -rf cache/*</pre>
                </li>
            </ol>
        </div>

        <!-- Actions rapides -->
        <div class="test-section">
            <h2>⚡ Actions Rapides</h2>
            <button class="button" onclick="location.reload()">🔄 Relancer le diagnostic</button>
            <button class="button" onclick="if(confirm('Supprimer ce fichier de diagnostic?')) { window.location='?delete=1'; }">🗑️ Supprimer ce fichier</button>
        </div>

        <?php
        // Option pour supprimer ce fichier
        if (isset($_GET['delete'])) {
            if (unlink(__FILE__)) {
                echo "<script>alert('Fichier supprimé avec succès'); window.location='/';</script>";
            } else {
                echo "<div class='alert alert-danger'>Impossible de supprimer le fichier. Supprimez-le manuellement!</div>";
            }
        }
        ?>
    </div>
</body>
</html>