<?php
/**
 * ╔══════════════════════════════════════════════════════════════╗
 * ║      CEDREPS ULTIMATE - Système d'Évaluation 4 Dimensions    ║
 * ║        Motricité - Intelligibilité - Sensibilité - Sociabilité         ║
 * ╚══════════════════════════════════════════════════════════════╝
 */

require_once 'config.php';
require_once 'functions.php';

if (!isAuthenticated()) {
    header('Location: login.php');
    exit;
}

$user = getCurrentUser();
$lessonId = $_GET['lesson'] ?? 0;
$classId = $_GET['class'] ?? 0;

// Récupération des données
$students = getClassStudents($classId);
$lesson = getLessonDetails($lessonId);
$socialRoles = getSocialRoles();
$evaluations = getExistingEvaluations($lessonId);
?>
<!DOCTYPE html>
<html lang="fr" data-theme="light">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Évaluation 4 Dimensions - CEDREPS Ultimate</title>
    
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script defer src="https://cdn.jsdelivr.net/npm/alpinejs@3.x.x/dist/cdn.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@100..900&family=Outfit:wght@100..900&display=swap');
        
        * { font-family: 'Inter', system-ui, sans-serif; }
        h1, h2, h3, h4, h5, h6 { font-family: 'Outfit', sans-serif; }
        
        .dimension-card {
            position: relative;
            overflow: hidden;
            transition: all 0.3s ease;
        }
        
        .dimension-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: var(--dimension-color);
        }
        
        .dimension-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
        }
        
        .slider-track {
            background: linear-gradient(90deg, 
                #EF4444 0%, 
                #F59E0B 25%, 
                #FCD34D 50%, 
                #84CC16 75%, 
                #10B981 100%
            );
        }
        
        .score-bubble {
            position: absolute;
            top: -40px;
            transform: translateX(-50%);
            background: var(--bubble-color);
            color: white;
            padding: 4px 12px;
            border-radius: 20px;
            font-weight: bold;
            font-size: 14px;
            white-space: nowrap;
            opacity: 0;
            transition: opacity 0.3s;
        }
        
        .slider:hover .score-bubble,
        .slider:focus .score-bubble {
            opacity: 1;
        }
        
        .student-card {
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .student-card:hover {
            transform: scale(1.02);
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
        }
        
        .student-card.selected {
            border-color: #3B82F6;
            background: linear-gradient(135deg, #EBF5FF 0%, #DBEAFE 100%);
        }
        
        .progress-ring {
            transform: rotate(-90deg);
        }
        
        .progress-ring__circle {
            stroke-dasharray: 251.2;
            stroke-dashoffset: 251.2;
            animation: progress 1s ease-out forwards;
        }
        
        @keyframes progress {
            to {
                stroke-dashoffset: var(--progress);
            }
        }
        
        .radar-chart-container {
            position: relative;
            height: 300px;
            width: 100%;
        }
        
        .evaluation-status {
            position: absolute;
            top: 10px;
            right: 10px;
            width: 12px;
            height: 12px;
            border-radius: 50%;
        }
        
        .status-not-evaluated { background: #EF4444; }
        .status-in-progress { background: #F59E0B; }
        .status-evaluated { background: #10B981; }
        
        .dimension-tooltip {
            position: absolute;
            background: #1F2937;
            color: white;
            padding: 8px 12px;
            border-radius: 8px;
            font-size: 12px;
            white-space: nowrap;
            z-index: 1000;
            pointer-events: none;
            opacity: 0;
            transition: opacity 0.3s;
        }
        
        .dimension-tooltip.show {
            opacity: 1;
        }
        
        .batch-action-bar {
            position: fixed;
            bottom: 20px;
            left: 50%;
            transform: translateX(-50%);
            background: white;
            padding: 16px 24px;
            border-radius: 16px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.2);
            z-index: 50;
            display: none;
        }
        
        .batch-action-bar.show {
            display: flex;
        }
    </style>
</head>
<body class="min-h-screen bg-gradient-to-br from-gray-50 to-gray-100" x-data="evaluationSystem">
    
    <!-- Header -->
    <header class="bg-white shadow-sm sticky top-0 z-40">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex items-center justify-between py-4">
                <div class="flex items-center space-x-4">
                    <a href="lessons.php?cycle=<?= $lesson['cycle_id'] ?>" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-arrow-left"></i>
                    </a>
                    <div>
                        <h1 class="text-2xl font-bold">Évaluation 4 Dimensions</h1>
                        <p class="text-sm text-gray-600">
                            <?= htmlspecialchars($lesson['title']) ?> - 
                            <?= htmlspecialchars($lesson['class_name']) ?>
                        </p>
                    </div>
                </div>
                
                <!-- Actions -->
                <div class="flex items-center space-x-4">
                    <button @click="viewMode = viewMode === 'grid' ? 'list' : 'grid'" 
                            class="text-gray-500 hover:text-gray-700">
                        <i class="fas" :class="viewMode === 'grid' ? 'fa-list' : 'fa-th'"></i>
                    </button>
                    <button @click="showHelp()" class="text-blue-500 hover:text-blue-600">
                        <i class="fas fa-question-circle"></i>
                    </button>
                    <button @click="saveAllEvaluations()" 
                            class="bg-gradient-to-r from-blue-500 to-purple-600 text-white px-4 py-2 rounded-lg hover:shadow-lg transition">
                        <i class="fas fa-save mr-2"></i> Sauvegarder
                    </button>
                </div>
            </div>
        </div>
        
        <!-- Progress Bar -->
        <div class="h-1 bg-gray-200">
            <div class="h-full bg-gradient-to-r from-blue-500 to-purple-600 transition-all duration-500" 
                 :style="`width: ${evaluationProgress}%`"></div>
        </div>
    </header>
    
    <main class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        
        <!-- Info Cards -->
        <div class="grid grid-cols-4 gap-4 mb-8">
            <div class="bg-white rounded-xl shadow-sm p-4">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm text-gray-500">Élèves évalués</p>
                        <p class="text-2xl font-bold text-gray-900">
                            <span x-text="evaluatedCount"></span> / <?= count($students) ?>
                        </p>
                    </div>
                    <div class="w-12 h-12 rounded-xl bg-blue-100 flex items-center justify-center text-blue-600">
                        <i class="fas fa-users"></i>
                    </div>
                </div>
            </div>
            
            <div class="bg-white rounded-xl shadow-sm p-4">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm text-gray-500">Moyenne Motricité</p>
                        <p class="text-2xl font-bold text-red-600" x-text="avgMotricite.toFixed(1)"></p>
                    </div>
                    <svg class="w-12 h-12">
                        <circle cx="24" cy="24" r="20" stroke="#FEE2E2" stroke-width="4" fill="none"/>
                        <circle class="progress-ring__circle" cx="24" cy="24" r="20" 
                                stroke="#EF4444" stroke-width="4" fill="none"
                                :style="`--progress: ${251.2 - (251.2 * avgMotricite / 20)}`"/>
                    </svg>
                </div>
            </div>
            
            <div class="bg-white rounded-xl shadow-sm p-4">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm text-gray-500">Moyenne Globale</p>
                        <p class="text-2xl font-bold text-green-600" x-text="avgGlobal.toFixed(1)"></p>
                    </div>
                    <svg class="w-12 h-12">
                        <circle cx="24" cy="24" r="20" stroke="#D1FAE5" stroke-width="4" fill="none"/>
                        <circle class="progress-ring__circle" cx="24" cy="24" r="20" 
                                stroke="#10B981" stroke-width="4" fill="none"
                                :style="`--progress: ${251.2 - (251.2 * avgGlobal / 20)}`"/>
                    </svg>
                </div>
            </div>
            
            <div class="bg-white rounded-xl shadow-sm p-4">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm text-gray-500">Temps estimé</p>
                        <p class="text-2xl font-bold text-gray-900">
                            <span x-text="Math.ceil(<?= count($students) ?> * 2)"></span> min
                        </p>
                    </div>
                    <div class="w-12 h-12 rounded-xl bg-purple-100 flex items-center justify-center text-purple-600">
                        <i class="fas fa-clock"></i>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Main Grid Layout -->
        <div class="grid grid-cols-3 gap-8">
            
            <!-- Left: Students List -->
            <div class="col-span-1">
                <div class="bg-white rounded-xl shadow-sm p-6">
                    <div class="flex items-center justify-between mb-4">
                        <h2 class="text-lg font-bold">Élèves</h2>
                        <div class="flex items-center space-x-2">
                            <button @click="selectAllStudents()" class="text-sm text-blue-600 hover:text-blue-700">
                                Tout sélectionner
                            </button>
                        </div>
                    </div>
                    
                    <!-- Search -->
                    <div class="relative mb-4">
                        <input type="search" 
                               x-model="searchStudent"
                               placeholder="Rechercher un élève..." 
                               class="w-full pl-10 pr-4 py-2 bg-gray-100 rounded-lg text-sm focus:outline-none focus:ring-2 focus:ring-blue-500">
                        <i class="fas fa-search absolute left-3 top-2.5 text-gray-400"></i>
                    </div>
                    
                    <!-- Students Grid/List -->
                    <div class="space-y-2 max-h-[600px] overflow-y-auto">
                        <?php foreach ($students as $student): ?>
                        <div @click="selectStudent(<?= $student['id'] ?>)" 
                             class="student-card bg-gray-50 rounded-lg p-3 flex items-center space-x-3"
                             :class="{ 'selected': selectedStudent === <?= $student['id'] ?> }">
                            <div class="evaluation-status" 
                                 :class="{
                                     'status-evaluated': evaluations[<?= $student['id'] ?>]?.complete,
                                     'status-in-progress': evaluations[<?= $student['id'] ?>] && !evaluations[<?= $student['id'] ?>].complete,
                                     'status-not-evaluated': !evaluations[<?= $student['id'] ?>]
                                 }"></div>
                            <img src="https://ui-avatars.com/api/?name=<?= urlencode($student['name']) ?>&background=3B82F6&color=ffffff" 
                                 class="w-10 h-10 rounded-full" alt="Avatar">
                            <div class="flex-1">
                                <p class="font-semibold text-sm"><?= htmlspecialchars($student['name']) ?></p>
                                <p class="text-xs text-gray-500">
                                    <?= htmlspecialchars($student['level']) ?>
                                    <?php if ($student['dispense']): ?>
                                    <span class="ml-2 text-red-500">
                                        <i class="fas fa-user-slash"></i> Dispensé
                                    </span>
                                    <?php endif; ?>
                                </p>
                            </div>
                            <div x-show="evaluations[<?= $student['id'] ?>]?.global_score" class="text-right">
                                <p class="text-lg font-bold" x-text="evaluations[<?= $student['id'] ?>]?.global_score"></p>
                                <p class="text-xs text-gray-500">/ 20</p>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
            
            <!-- Center: Evaluation Form -->
            <div class="col-span-2">
                <div x-show="selectedStudent" class="space-y-6">
                    
                    <!-- Student Header -->
                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <div class="flex items-center justify-between mb-4">
                            <div class="flex items-center space-x-4">
                                <img :src="`https://ui-avatars.com/api/?name=${encodeURIComponent(getStudentName(selectedStudent))}&background=3B82F6&color=ffffff`" 
                                     class="w-16 h-16 rounded-full" alt="Avatar">
                                <div>
                                    <h3 class="text-xl font-bold" x-text="getStudentName(selectedStudent)"></h3>
                                    <p class="text-sm text-gray-600">Évaluation pour la leçon <?= $lesson['lesson_number'] ?></p>
                                </div>
                            </div>
                            <div class="flex items-center space-x-2">
                                <button @click="previousStudent()" 
                                        class="p-2 text-gray-500 hover:text-gray-700">
                                    <i class="fas fa-chevron-left"></i>
                                </button>
                                <span class="text-sm text-gray-500">
                                    <span x-text="getStudentIndex(selectedStudent) + 1"></span> / <?= count($students) ?>
                                </span>
                                <button @click="nextStudent()" 
                                        class="p-2 text-gray-500 hover:text-gray-700">
                                    <i class="fas fa-chevron-right"></i>
                                </button>
                            </div>
                        </div>
                        
                        <!-- Quick Actions -->
                        <div class="flex items-center space-x-2">
                            <button @click="quickScore('absent')" 
                                    class="px-3 py-1 bg-gray-100 text-gray-700 rounded-lg text-sm hover:bg-gray-200">
                                <i class="fas fa-user-times mr-1"></i> Absent
                            </button>
                            <button @click="quickScore('dispense')" 
                                    class="px-3 py-1 bg-red-100 text-red-700 rounded-lg text-sm hover:bg-red-200">
                                <i class="fas fa-user-slash mr-1"></i> Dispensé
                            </button>
                            <button @click="quickScore('excellent')" 
                                    class="px-3 py-1 bg-green-100 text-green-700 rounded-lg text-sm hover:bg-green-200">
                                <i class="fas fa-star mr-1"></i> Excellent (18+)
                            </button>
                            <button @click="quickScore('good')" 
                                    class="px-3 py-1 bg-blue-100 text-blue-700 rounded-lg text-sm hover:bg-blue-200">
                                <i class="fas fa-thumbs-up mr-1"></i> Bien (14-17)
                            </button>
                        </div>
                    </div>
                    
                    <!-- 4 Dimensions Evaluation -->
                    <div class="grid grid-cols-2 gap-6">
                        
                        <!-- Motricité -->
                        <div class="dimension-card bg-white rounded-xl shadow-sm p-6" 
                             style="--dimension-color: #EF4444">
                            <div class="flex items-center justify-between mb-4">
                                <div>
                                    <h4 class="font-bold text-lg">Motricité</h4>
                                    <p class="text-xs text-gray-500">Corps vital, transformations motrices</p>
                                </div>
                                <div class="w-12 h-12 rounded-xl bg-red-100 flex items-center justify-center text-red-600">
                                    <i class="fas fa-running"></i>
                                </div>
                            </div>
                            
                            <div class="relative mb-4">
                                <input type="range" 
                                       min="0" max="20" step="0.5"
                                       x-model="currentEvaluation.motricite"
                                       class="w-full slider slider-track"
                                       style="--bubble-color: #EF4444">
                                <div class="score-bubble" 
                                     :style="`left: ${currentEvaluation.motricite * 5}%`"
                                     x-text="currentEvaluation.motricite"></div>
                            </div>
                            
                            <div class="flex justify-between text-xs text-gray-500">
                                <span>Non acquis</span>
                                <span>En cours</span>
                                <span>Acquis</span>
                                <span>Expert</span>
                            </div>
                            
                            <!-- Critères détaillés -->
                            <div class="mt-4 space-y-2">
                                <label class="flex items-center text-xs">
                                    <input type="checkbox" class="mr-2 text-red-600">
                                    <span>Coordination efficace</span>
                                </label>
                                <label class="flex items-center text-xs">
                                    <input type="checkbox" class="mr-2 text-red-600">
                                    <span>Vitesse d'exécution</span>
                                </label>
                                <label class="flex items-center text-xs">
                                    <input type="checkbox" class="mr-2 text-red-600">
                                    <span>Précision gestuelle</span>
                                </label>
                            </div>
                        </div>
                        
                        <!-- Intelligibilité -->
                        <div class="dimension-card bg-white rounded-xl shadow-sm p-6" 
                             style="--dimension-color: #3B82F6">
                            <div class="flex items-center justify-between mb-4">
                                <div>
                                    <h4 class="font-bold text-lg">Intelligibilité</h4>
                                    <p class="text-xs text-gray-500">Compréhension, analyse, stratégies</p>
                                </div>
                                <div class="w-12 h-12 rounded-xl bg-blue-100 flex items-center justify-center text-blue-600">
                                    <i class="fas fa-brain"></i>
                                </div>
                            </div>
                            
                            <div class="relative mb-4">
                                <input type="range" 
                                       min="0" max="20" step="0.5"
                                       x-model="currentEvaluation.intelligibilite"
                                       class="w-full slider slider-track"
                                       style="--bubble-color: #3B82F6">
                                <div class="score-bubble" 
                                     :style="`left: ${currentEvaluation.intelligibilite * 5}%`"
                                     x-text="currentEvaluation.intelligibilite"></div>
                            </div>
                            
                            <div class="flex justify-between text-xs text-gray-500">
                                <span>Non acquis</span>
                                <span>En cours</span>
                                <span>Acquis</span>
                                <span>Expert</span>
                            </div>
                            
                            <!-- Critères détaillés -->
                            <div class="mt-4 space-y-2">
                                <label class="flex items-center text-xs">
                                    <input type="checkbox" class="mr-2 text-blue-600">
                                    <span>Compréhension des consignes</span>
                                </label>
                                <label class="flex items-center text-xs">
                                    <input type="checkbox" class="mr-2 text-blue-600">
                                    <span>Analyse de la situation</span>
                                </label>
                                <label class="flex items-center text-xs">
                                    <input type="checkbox" class="mr-2 text-blue-600">
                                    <span>Adaptation stratégique</span>
                                </label>
                            </div>
                        </div>
                        
                        <!-- Sensibilité -->
                        <div class="dimension-card bg-white rounded-xl shadow-sm p-6" 
                             style="--dimension-color: #F59E0B">
                            <div class="flex items-center justify-between mb-4">
                                <div>
                                    <h4 class="font-bold text-lg">Sensibilité</h4>
                                    <p class="text-xs text-gray-500">Ressentis, émotions, rapport au corps</p>
                                </div>
                                <div class="w-12 h-12 rounded-xl bg-yellow-100 flex items-center justify-center text-yellow-600">
                                    <i class="fas fa-heart"></i>
                                </div>
                            </div>
                            
                            <div class="relative mb-4">
                                <input type="range" 
                                       min="0" max="20" step="0.5"
                                       x-model="currentEvaluation.sensibilite"
                                       class="w-full slider slider-track"
                                       style="--bubble-color: #F59E0B">
                                <div class="score-bubble" 
                                     :style="`left: ${currentEvaluation.sensibilite * 5}%`"
                                     x-text="currentEvaluation.sensibilite"></div>
                            </div>
                            
                            <div class="flex justify-between text-xs text-gray-500">
                                <span>Non acquis</span>
                                <span>En cours</span>
                                <span>Acquis</span>
                                <span>Expert</span>
                            </div>
                            
                            <!-- Critères détaillés -->
                            <div class="mt-4 space-y-2">
                                <label class="flex items-center text-xs">
                                    <input type="checkbox" class="mr-2 text-yellow-600">
                                    <span>Gestion des émotions</span>
                                </label>
                                <label class="flex items-center text-xs">
                                    <input type="checkbox" class="mr-2 text-yellow-600">
                                    <span>Écoute de ses sensations</span>
                                </label>
                                <label class="flex items-center text-xs">
                                    <input type="checkbox" class="mr-2 text-yellow-600">
                                    <span>Expression corporelle</span>
                                </label>
                            </div>
                        </div>
                        
                        <!-- Sociabilité -->
                        <div class="dimension-card bg-white rounded-xl shadow-sm p-6" 
                             style="--dimension-color: #10B981">
                            <div class="flex items-center justify-between mb-4">
                                <div>
                                    <h4 class="font-bold text-lg">Sociabilité</h4>
                                    <p class="text-xs text-gray-500">Rapport aux autres, coopération</p>
                                </div>
                                <div class="w-12 h-12 rounded-xl bg-green-100 flex items-center justify-center text-green-600">
                                    <i class="fas fa-users"></i>
                                </div>
                            </div>
                            
                            <div class="relative mb-4">
                                <input type="range" 
                                       min="0" max="20" step="0.5"
                                       x-model="currentEvaluation.sociabilite"
                                       class="w-full slider slider-track"
                                       style="--bubble-color: #10B981">
                                <div class="score-bubble" 
                                     :style="`left: ${currentEvaluation.sociabilite * 5}%`"
                                     x-text="currentEvaluation.sociabilite"></div>
                            </div>
                            
                            <div class="flex justify-between text-xs text-gray-500">
                                <span>Non acquis</span>
                                <span>En cours</span>
                                <span>Acquis</span>
                                <span>Expert</span>
                            </div>
                            
                            <!-- Critères détaillés -->
                            <div class="mt-4 space-y-2">
                                <label class="flex items-center text-xs">
                                    <input type="checkbox" class="mr-2 text-green-600">
                                    <span>Coopération active</span>
                                </label>
                                <label class="flex items-center text-xs">
                                    <input type="checkbox" class="mr-2 text-green-600">
                                    <span>Communication efficace</span>
                                </label>
                                <label class="flex items-center text-xs">
                                    <input type="checkbox" class="mr-2 text-green-600">
                                    <span>Respect des règles</span>
                                </label>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Additional Info -->
                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <div class="grid grid-cols-2 gap-6">
                            <!-- Social Role -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">
                                    Rôle social dans la leçon
                                </label>
                                <select x-model="currentEvaluation.social_role" 
                                        class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                                    <option value="">-- Aucun rôle spécifique --</option>
                                    <?php foreach ($socialRoles as $role): ?>
                                    <option value="<?= $role['name'] ?>">
                                        <?= htmlspecialchars($role['name']) ?> - <?= htmlspecialchars($role['description']) ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <!-- Progress Indicator -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">
                                    Indicateur de progression
                                </label>
                                <div class="flex space-x-2">
                                    <button @click="currentEvaluation.progress_indicator = 'en_progres'"
                                            :class="currentEvaluation.progress_indicator === 'en_progres' ? 'bg-yellow-500 text-white' : 'bg-gray-100 text-gray-700'"
                                            class="flex-1 px-3 py-2 rounded-lg text-sm font-medium transition">
                                        <i class="fas fa-spinner mr-1"></i> En progrès
                                    </button>
                                    <button @click="currentEvaluation.progress_indicator = 'acquis'"
                                            :class="currentEvaluation.progress_indicator === 'acquis' ? 'bg-green-500 text-white' : 'bg-gray-100 text-gray-700'"
                                            class="flex-1 px-3 py-2 rounded-lg text-sm font-medium transition">
                                        <i class="fas fa-check mr-1"></i> Acquis
                                    </button>
                                    <button @click="currentEvaluation.progress_indicator = 'expert'"
                                            :class="currentEvaluation.progress_indicator === 'expert' ? 'bg-purple-500 text-white' : 'bg-gray-100 text-gray-700'"
                                            class="flex-1 px-3 py-2 rounded-lg text-sm font-medium transition">
                                        <i class="fas fa-star mr-1"></i> Expert
                                    </button>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Remarks -->
                        <div class="mt-6">
                            <label class="block text-sm font-medium text-gray-700 mb-2">
                                Remarques et observations
                            </label>
                            <textarea x-model="currentEvaluation.remarks" 
                                      rows="3" 
                                      placeholder="Points forts, axes d'amélioration, comportement..."
                                      class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"></textarea>
                        </div>
                        
                        <!-- Score Summary -->
                        <div class="mt-6 p-4 bg-gradient-to-r from-blue-50 to-purple-50 rounded-lg">
                            <div class="flex items-center justify-between">
                                <div>
                                    <h4 class="font-bold text-lg">Score Global</h4>
                                    <p class="text-sm text-gray-600">Moyenne des 4 dimensions</p>
                                </div>
                                <div class="text-right">
                                    <p class="text-4xl font-bold" 
                                       :class="{
                                           'text-red-600': globalScore < 10,
                                           'text-yellow-600': globalScore >= 10 && globalScore < 14,
                                           'text-green-600': globalScore >= 14
                                       }"
                                       x-text="globalScore.toFixed(1)"></p>
                                    <p class="text-sm text-gray-500">/ 20</p>
                                </div>
                            </div>
                            
                            <!-- Radar Chart -->
                            <div class="mt-4">
                                <canvas id="studentRadarChart" class="radar-chart-container"></canvas>
                            </div>
                        </div>
                        
                        <!-- Action Buttons -->
                        <div class="mt-6 flex justify-between">
                            <button @click="resetCurrentEvaluation()" 
                                    class="px-4 py-2 bg-gray-100 text-gray-700 rounded-lg hover:bg-gray-200 transition">
                                <i class="fas fa-undo mr-2"></i> Réinitialiser
                            </button>
                            <div class="flex space-x-3">
                                <button @click="saveAndContinue()" 
                                        class="px-4 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition">
                                    <i class="fas fa-save mr-2"></i> Sauvegarder et continuer
                                </button>
                                <button @click="saveAndNext()" 
                                        class="px-4 py-2 bg-gradient-to-r from-blue-500 to-purple-600 text-white rounded-lg hover:shadow-lg transition">
                                    <i class="fas fa-arrow-right mr-2"></i> Suivant
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Empty State -->
                <div x-show="!selectedStudent" class="bg-white rounded-xl shadow-sm p-12 text-center">
                    <i class="fas fa-user-check text-6xl text-gray-300 mb-4"></i>
                    <h3 class="text-xl font-semibold text-gray-700 mb-2">Sélectionnez un élève</h3>
                    <p class="text-gray-500">Cliquez sur un élève dans la liste pour commencer l'évaluation</p>
                </div>
            </div>
        </div>
    </main>
    
    <!-- Batch Action Bar -->
    <div class="batch-action-bar" :class="{ 'show': selectedStudents.length > 1 }">
        <div class="flex items-center space-x-4">
            <p class="text-sm font-medium">
                <span x-text="selectedStudents.length"></span> élèves sélectionnés
            </p>
            <button @click="applyBatchScore()" 
                    class="px-4 py-2 bg-blue-500 text-white rounded-lg text-sm hover:bg-blue-600 transition">
                <i class="fas fa-magic mr-1"></i> Appliquer une note
            </button>
            <button @click="clearSelection()" 
                    class="text-gray-500 hover:text-gray-700">
                <i class="fas fa-times"></i>
            </button>
        </div>
    </div>
    
    <!-- JavaScript -->
    <script>
        // Students Data from PHP
        const studentsData = <?= json_encode($students) ?>;
        const existingEvaluations = <?= json_encode($evaluations) ?>;
        
        // Alpine.js Component
        document.addEventListener('alpine:init', () => {
            Alpine.data('evaluationSystem', () => ({
                students: studentsData,
                selectedStudent: null,
                selectedStudents: [],
                searchStudent: '',
                viewMode: 'grid',
                evaluations: existingEvaluations || {},
                currentEvaluation: {
                    motricite: 10,
                    intelligibilite: 10,
                    sensibilite: 10,
                    sociabilite: 10,
                    social_role: '',
                    progress_indicator: 'en_progres',
                    remarks: ''
                },
                radarChart: null,
                
                get evaluatedCount() {
                    return Object.keys(this.evaluations).length;
                },
                
                get evaluationProgress() {
                    return (this.evaluatedCount / this.students.length) * 100;
                },
                
                get avgMotricite() {
                    const values = Object.values(this.evaluations).map(e => parseFloat(e.motricite) || 0);
                    return values.length ? values.reduce((a, b) => a + b) / values.length : 0;
                },
                
                get avgGlobal() {
                    const values = Object.values(this.evaluations).map(e => parseFloat(e.global_score) || 0);
                    return values.length ? values.reduce((a, b) => a + b) / values.length : 0;
                },
                
                get globalScore() {
                    return (
                        parseFloat(this.currentEvaluation.motricite) +
                        parseFloat(this.currentEvaluation.intelligibilite) +
                        parseFloat(this.currentEvaluation.sensibilite) +
                        parseFloat(this.currentEvaluation.sociabilite)
                    ) / 4;
                },
                
                init() {
                    this.initRadarChart();
                    if (this.students.length > 0 && !this.selectedStudent) {
                        this.selectStudent(this.students[0].id);
                    }
                },
                
                selectStudent(studentId) {
                    this.selectedStudent = studentId;
                    
                    // Load existing evaluation if exists
                    if (this.evaluations[studentId]) {
                        this.currentEvaluation = { ...this.evaluations[studentId] };
                    } else {
                        this.resetCurrentEvaluation();
                    }
                    
                    this.updateRadarChart();
                },
                
                getStudentName(studentId) {
                    const student = this.students.find(s => s.id === studentId);
                    return student ? student.name : '';
                },
                
                getStudentIndex(studentId) {
                    return this.students.findIndex(s => s.id === studentId);
                },
                
                previousStudent() {
                    const currentIndex = this.getStudentIndex(this.selectedStudent);
                    if (currentIndex > 0) {
                        this.selectStudent(this.students[currentIndex - 1].id);
                    }
                },
                
                nextStudent() {
                    const currentIndex = this.getStudentIndex(this.selectedStudent);
                    if (currentIndex < this.students.length - 1) {
                        this.selectStudent(this.students[currentIndex + 1].id);
                    }
                },
                
                quickScore(type) {
                    switch(type) {
                        case 'absent':
                            this.currentEvaluation = {
                                motricite: 0,
                                intelligibilite: 0,
                                sensibilite: 0,
                                sociabilite: 0,
                                social_role: '',
                                progress_indicator: 'en_progres',
                                remarks: 'Absent'
                            };
                            break;
                        case 'dispense':
                            this.currentEvaluation = {
                                motricite: 0,
                                intelligibilite: 0,
                                sensibilite: 0,
                                sociabilite: 0,
                                social_role: '',
                                progress_indicator: 'en_progres',
                                remarks: 'Dispensé'
                            };
                            break;
                        case 'excellent':
                            this.currentEvaluation = {
                                motricite: 18,
                                intelligibilite: 18,
                                sensibilite: 18,
                                sociabilite: 18,
                                social_role: this.currentEvaluation.social_role,
                                progress_indicator: 'expert',
                                remarks: 'Excellent travail'
                            };
                            break;
                        case 'good':
                            this.currentEvaluation = {
                                motricite: 15,
                                intelligibilite: 15,
                                sensibilite: 15,
                                sociabilite: 15,
                                social_role: this.currentEvaluation.social_role,
                                progress_indicator: 'acquis',
                                remarks: 'Bon travail'
                            };
                            break;
                    }
                    this.updateRadarChart();
                },
                
                resetCurrentEvaluation() {
                    this.currentEvaluation = {
                        motricite: 10,
                        intelligibilite: 10,
                        sensibilite: 10,
                        sociabilite: 10,
                        social_role: '',
                        progress_indicator: 'en_progres',
                        remarks: ''
                    };
                    this.updateRadarChart();
                },
                
                saveAndContinue() {
                    this.saveCurrentEvaluation();
                    Swal.fire({
                        icon: 'success',
                        title: 'Enregistré',
                        text: 'Évaluation sauvegardée',
                        timer: 1500,
                        showConfirmButton: false
                    });
                },
                
                saveAndNext() {
                    this.saveCurrentEvaluation();
                    this.nextStudent();
                },
                
                saveCurrentEvaluation() {
                    if (!this.selectedStudent) return;
                    
                    this.evaluations[this.selectedStudent] = {
                        ...this.currentEvaluation,
                        global_score: this.globalScore,
                        complete: true
                    };
                },
                
                async saveAllEvaluations() {
                    // Save current evaluation first
                    if (this.selectedStudent) {
                        this.saveCurrentEvaluation();
                    }
                    
                    // Send to server
                    const result = await Swal.fire({
                        title: 'Sauvegarder toutes les évaluations ?',
                        text: `${this.evaluatedCount} élèves évalués`,
                        icon: 'question',
                        showCancelButton: true,
                        confirmButtonColor: '#3B82F6',
                        cancelButtonColor: '#6B7280',
                        confirmButtonText: 'Sauvegarder',
                        cancelButtonText: 'Annuler'
                    });
                    
                    if (result.isConfirmed) {
                        // Ici, envoyer les données au serveur via AJAX
                        Swal.fire({
                            icon: 'success',
                            title: 'Sauvegardé !',
                            text: 'Toutes les évaluations ont été enregistrées',
                            confirmButtonColor: '#3B82F6'
                        });
                    }
                },
                
                initRadarChart() {
                    const ctx = document.getElementById('studentRadarChart');
                    if (!ctx) return;
                    
                    this.radarChart = new Chart(ctx, {
                        type: 'radar',
                        data: {
                            labels: ['Motricité', 'Intelligibilité', 'Sensibilité', 'Sociabilité'],
                            datasets: [{
                                label: 'Évaluation',
                                data: [10, 10, 10, 10],
                                backgroundColor: 'rgba(59, 130, 246, 0.2)',
                                borderColor: 'rgba(59, 130, 246, 1)',
                                pointBackgroundColor: 'rgba(59, 130, 246, 1)',
                                pointBorderColor: '#fff',
                                pointHoverBackgroundColor: '#fff',
                                pointHoverBorderColor: 'rgba(59, 130, 246, 1)'
                            }]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            scales: {
                                r: {
                                    beginAtZero: true,
                                    max: 20,
                                    ticks: {
                                        stepSize: 5
                                    }
                                }
                            },
                            plugins: {
                                legend: {
                                    display: false
                                }
                            }
                        }
                    });
                },
                
                updateRadarChart() {
                    if (!this.radarChart) return;
                    
                    this.radarChart.data.datasets[0].data = [
                        this.currentEvaluation.motricite,
                        this.currentEvaluation.intelligibilite,
                        this.currentEvaluation.sensibilite,
                        this.currentEvaluation.sociabilite
                    ];
                    
                    this.radarChart.update();
                },
                
                showHelp() {
                    Swal.fire({
                        title: 'Guide d\'évaluation CEDREPS',
                        html: `
                            <div class="text-left">
                                <h4 class="font-bold mb-2">Les 4 Dimensions</h4>
                                <ul class="text-sm space-y-2">
                                    <li><strong>Motricité :</strong> Le corps vital, les transformations motrices</li>
                                    <li><strong>Intelligibilité :</strong> Compréhension, analyse, stratégies</li>
                                    <li><strong>Sensibilité :</strong> Ressentis, émotions, rapport au corps</li>
                                    <li><strong>Sociabilité :</strong> Rapport aux autres, coopération</li>
                                </ul>
                                <h4 class="font-bold mt-4 mb-2">Échelle de notation</h4>
                                <ul class="text-sm space-y-1">
                                    <li>0-5 : Non acquis</li>
                                    <li>6-10 : En cours d'acquisition</li>
                                    <li>11-15 : Acquis</li>
                                    <li>16-20 : Expert</li>
                                </ul>
                            </div>
                        `,
                        icon: 'info',
                        confirmButtonColor: '#3B82F6'
                    });
                },
                
                selectAllStudents() {
                    this.selectedStudents = this.students.map(s => s.id);
                },
                
                clearSelection() {
                    this.selectedStudents = [];
                }
            }));
        });
    </script>
</body>
</html>