<?php
/**
 * ╔══════════════════════════════════════════════════════════════╗
 * ║         FONCTIONS ESSENTIELLES - CEDREPS ULTIMATE             ║
 * ║                   Version simplifiée                          ║
 * ╚══════════════════════════════════════════════════════════════╝
 */

// ===================================================================
// FONCTIONS DE SESSION ET AUTHENTIFICATION
// ===================================================================

/**
 * Vérifier si l'utilisateur est connecté
 */
function isAuthenticated() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    
    return isset($_SESSION['user_id']) && isset($_SESSION['user_email']);
}

/**
 * Obtenir l'utilisateur connecté
 */
function getCurrentUser() {
    if (!isAuthenticated()) {
        return null;
    }
    
    return [
        'id' => $_SESSION['user_id'] ?? null,
        'email' => $_SESSION['user_email'] ?? null,
        'name' => $_SESSION['user_name'] ?? 'Utilisateur',
    ];
}

/**
 * Déconnecter l'utilisateur
 */
function logout() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    
    $_SESSION = [];
    
    if (ini_get("session.use_cookies")) {
        $params = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000,
            $params["path"], $params["domain"],
            $params["secure"], $params["httponly"]
        );
    }
    
    session_destroy();
}

// ===================================================================
// FONCTIONS DE BASE DE DONNÉES
// ===================================================================

/**
 * Obtenir la connexion PDO
 */
if (!function_exists('getDB')) {
function getDB() {
    global $pdo;
    
    if ($pdo) {
        return $pdo;
    }
    
    // Si pas de connexion globale, essayer de se connecter
    if (file_exists(__DIR__ . '/config.php')) {
        require_once __DIR__ . '/config.php';
        return $pdo ?? null;
    }
    
    return null;
}
}

/**
 * Obtenir les classes d'un enseignant
 */
function getTeacherClasses($teacher_id) {
    $db = getDB();
    if (!$db) return [];
    
    try {
        $stmt = $db->prepare("
            SELECT * FROM classes 
            WHERE teacher_id = ? 
            AND is_active = 1 
            ORDER BY name
        ");
        $stmt->execute([$teacher_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return [];
    }
}

/**
 * Obtenir les élèves d'une classe
 */
function getClassStudents($class_id) {
    $db = getDB();
    if (!$db) return [];
    
    try {
        $stmt = $db->prepare("
            SELECT * FROM students 
            WHERE class_id = ? 
            ORDER BY name
        ");
        $stmt->execute([$class_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return [];
    }
}

/**
 * Obtenir les cycles d'une classe
 */
function getClassCycles($class_id) {
    $db = getDB();
    if (!$db) return [];
    
    try {
        $stmt = $db->prepare("
            SELECT * FROM cycles 
            WHERE class_id = ? 
            ORDER BY start_date DESC
        ");
        $stmt->execute([$class_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return [];
    }
}

/**
 * Obtenir les leçons d'un cycle
 */
function getCycleLessons($cycle_id) {
    $db = getDB();
    if (!$db) return [];
    
    try {
        $stmt = $db->prepare("
            SELECT * FROM lessons 
            WHERE cycle_id = ? 
            ORDER BY lesson_number
        ");
        $stmt->execute([$cycle_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return [];
    }
}

// ===================================================================
// FONCTIONS UTILITAIRES
// ===================================================================

/**
 * Sécuriser une chaîne pour l'affichage
 */
function e($string) {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

/**
 * Rediriger vers une URL
 */
function redirect($url) {
    header("Location: $url");
    exit;
}

/**
 * Obtenir la valeur d'un paramètre GET/POST
 */
function getParam($name, $default = null) {
    return $_REQUEST[$name] ?? $default;
}

/**
 * Vérifier si la requête est en AJAX
 */
function isAjax() {
    return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
           strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest';
}

/**
 * Envoyer une réponse JSON
 */
function jsonResponse($data, $status = 200) {
    http_response_code($status);
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}

/**
 * Générer un token CSRF
 */
function generateCSRFToken() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    
    return $_SESSION['csrf_token'];
}

/**
 * Vérifier un token CSRF
 */
function verifyCSRFToken($token) {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    
    return isset($_SESSION['csrf_token']) && 
           hash_equals($_SESSION['csrf_token'], $token);
}

// ===================================================================
// FONCTIONS SPÉCIFIQUES CEDREPS
// ===================================================================

/**
 * Obtenir les types d'ECC
 */
function getECCTypes() {
    return [
        'ECC1' => 'Produire une performance optimale',
        'ECC2' => 'Adapter ses déplacements',
        'ECC3' => 'S\'exprimer devant les autres',
        'ECC4' => 'Conduire et maîtriser un affrontement',
        'ECC5' => 'Réaliser une activité physique pour développer ses ressources',
        'ECC6' => 'S\'engager lucidement dans la pratique'
    ];
}

/**
 * Obtenir les rôles sociaux
 */
function getSocialRoles() {
    return [
        ['name' => 'Observateur', 'description' => 'Observer et analyser'],
        ['name' => 'Arbitre', 'description' => 'Faire respecter les règles'],
        ['name' => 'Coach', 'description' => 'Conseiller et encourager'],
        ['name' => 'Organisateur', 'description' => 'Organiser et gérer'],
        ['name' => 'Capitaine', 'description' => 'Diriger et motiver']
    ];
}

/**
 * Obtenir les détails d'une leçon
 */
function getLessonDetails($lesson_id) {
    $db = getDB();
    if (!$db) return null;
    
    try {
        $stmt = $db->prepare("
            SELECT l.*, c.name as cycle_name, c.ppsad, cl.name as class_name
            FROM lessons l
            LEFT JOIN cycles c ON l.cycle_id = c.id
            LEFT JOIN classes cl ON c.class_id = cl.id
            WHERE l.id = ?
        ");
        $stmt->execute([$lesson_id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return null;
    }
}

/**
 * Obtenir les évaluations existantes
 */
function getExistingEvaluations($lesson_id) {
    $db = getDB();
    if (!$db) return [];
    
    try {
        $stmt = $db->prepare("
            SELECT * FROM evaluations 
            WHERE lesson_id = ?
        ");
        $stmt->execute([$lesson_id]);
        
        $evaluations = [];
        while ($eval = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $evaluations[$eval['student_id']] = $eval;
        }
        
        return $evaluations;
    } catch (Exception $e) {
        return [];
    }
}

// ===================================================================
// INITIALISATION
// ===================================================================

// Démarrer la session si pas déjà fait
if (session_status() === PHP_SESSION_NONE && PHP_SAPI !== 'cli') {
    session_start();
}

// Inclure la config si elle existe et n'est pas déjà chargée
if (!isset($pdo) && file_exists(__DIR__ . '/config.php')) {
    require_once __DIR__ . '/config.php';
}
?>