# Guide d'Enrichissement : Intégration Avancée des Concepts CEDREPS

## 🎯 Vision

Transformer votre application EPS en **plateforme de référence CEDREPS** en intégrant complètement le cadre théorique des documents analysés.

---

## 📋 Plan d'Action en 10 Étapes

### Phase 1 : Enrichissement Conceptuel Immédiat

#### 1. **Base de Données des 6 ECC**

**Ajout SQL :**
```sql
-- Table des Expériences Corporelles et Culturelles
CREATE TABLE IF NOT EXISTS ecc_definitions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    code VARCHAR(10) NOT NULL, -- ECC1 à ECC6
    name VARCHAR(100) NOT NULL,
    imaginaire_symbolique TEXT,
    but_activite TEXT,
    mobiles_typiques TEXT,
    icon VARCHAR(50),
    color VARCHAR(7), -- Code couleur hexadécimal
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Données initiales basées sur le Cahier 18
INSERT INTO ecc_definitions (code, name, imaginaire_symbolique, but_activite, mobiles_typiques, icon, color) VALUES
('ECC1', 'Performance', 
 'Dépassement de soi, conquête héroïque de l\'espace-temps, domination éphémère sur le monde',
 'Produire une performance mesurable dans un cadre normé',
 'Se dépasser, battre son record, explorer ses limites',
 'fa-trophy', '#FF6B6B'),

('ECC2', 'Aventure',
 'Explorer l\'inconnu, vivre l\'inattendu, devenir le héros de son épopée',
 'Conduire et adapter son déplacement dans des environnements variés/incertains',
 'Découvrir, s\'aventurer, être autonome dans le milieu',
 'fa-compass', '#4ECDC4'),

('ECC3', 'Arts du mouvement',
 'Expression de soi, création, communication sensible',
 'Créer et présenter une composition corporelle porteuse de sens',
 'S\'exprimer, créer, émouvoir, partager',
 'fa-theater-masks', '#95E77E'),

('ECC4', 'Confrontation à l\'autre',
 'Affrontement codifié, défi mutuel, reconnaissance dans l\'opposition',
 'S\'opposer individuellement ou collectivement pour gagner',
 'Gagner, coopérer, élaborer des stratégies',
 'fa-handshake', '#FFE66D'),

('ECC5', 'Découverte et soin de soi',
 'Bien-être, transformation de soi, harmonie corps-esprit',
 'Développer ses ressources pour agir sur soi de façon éclairée',
 'Se connaître, se transformer, prendre soin de soi',
 'fa-heart', '#A8DADC'),

('ECC6', 'Acrobatie',
 'Défier la gravité et l\'équilibre, jouer avec le vertige',
 'Réaliser des actions motrices défiant les lois habituelles',
 'Oser, maîtriser le risque, impressionner',
 'fa-star', '#F1FAEE');
```

#### 2. **Enrichir la Table Cycles avec le Fonds Culturel**

```sql
ALTER TABLE cycles 
ADD COLUMN fonds_culturel_mobiles TEXT COMMENT 'Mobiles du pratiquant cultivé',
ADD COLUMN fonds_culturel_principes TEXT COMMENT 'Principes organisateurs de l\'activité',
ADD COLUMN fonds_culturel_rapport_autre TEXT COMMENT 'Rapport structurant à l\'autre',
ADD COLUMN contraintes_emblematiques TEXT COMMENT 'Contraintes qui caractérisent la FPS';
```

#### 3. **Créer une Bibliothèque d'Objets d'Enseignement**

```sql
-- Référentiel des OE par PPSAD et niveau
CREATE TABLE IF NOT EXISTS objets_enseignement_ref (
    id INT AUTO_INCREMENT PRIMARY KEY,
    ppsad VARCHAR(100) NOT NULL,
    ecc_type VARCHAR(10),
    niveau_scolaire ENUM('cycle3', 'cycle4', 'lycee') NOT NULL,
    objet_enseignement TEXT NOT NULL,
    pas_en_avant TEXT COMMENT 'Le saut qualitatif visé',
    prerequis TEXT,
    exemples_fps TEXT,
    ordre_progression INT,
    source_cahier VARCHAR(100) COMMENT 'Référence au cahier CEDREPS',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_ppsad_niveau (ppsad, niveau_scolaire)
);

-- Exemples tirés des documents
INSERT INTO objets_enseignement_ref (ppsad, ecc_type, niveau_scolaire, objet_enseignement, pas_en_avant, exemples_fps, ordre_progression, source_cahier) VALUES
('Natation', 'ECC1', 'cycle3', 
 'De la fréquence CIBLE vers l\'amplitude',
 'Passer d\'une nage en fréquence à une nage en amplitude pour être plus efficace',
 'Le 12 secondes', 1, 'Cahier 15'),
 
('Tennis de table', 'ECC4', 'cycle3',
 'Jouer tout en coup droit',
 'Maîtriser le placement et le déplacement pour jouer uniquement en coup droit',
 'Tout en CD', 1, 'Cahier 15'),
 
('Handball', 'ECC4', 'cycle4',
 'Le jeu en mouvement individuel',
 'Passer d\'un jeu statique à un jeu en mouvement permanent',
 'Jeu en mouvement', 2, 'Cahier 15'),
 
('Escalade', 'ECC2', 'cycle4',
 'La mouli-vache',
 'Apprendre à grimper et assurer en moulinette de façon autonome',
 'Moulivache', 1, 'Cahier 16');
```

---

### Phase 2 : Interface Utilisateur Enrichie

#### 4. **Dashboard Amélioré avec Vue ECC**

**Nouveau fichier : `dashboard_ecc.php`**
```php
<?php
require_once 'functions.php';
if (!isLoggedIn()) redirect('login.php');

$user = currentUser($pdo);

// Récupérer les cycles par ECC
$stmt = $pdo->prepare('
    SELECT 
        c.ecc_type,
        ecc.name as ecc_name,
        ecc.color,
        ecc.icon,
        COUNT(DISTINCT cy.id) as nb_cycles,
        COUNT(DISTINCT l.id) as nb_lessons
    FROM classes cl
    LEFT JOIN cycles cy ON cl.id = cy.class_id
    LEFT JOIN lessons l ON cy.id = l.cycle_id
    LEFT JOIN ecc_definitions ecc ON cy.ecc_type = ecc.code
    WHERE cl.teacher_id = ?
    GROUP BY c.ecc_type
');
$stmt->execute([$user['id']]);
$ecc_stats = $stmt->fetchAll();

include 'header.php';
?>

<div class="container mt-4">
    <h2>Tableau de Bord - Vue par Expériences Corporelles et Culturelles</h2>
    
    <div class="row mt-4">
        <?php foreach ($ecc_stats as $ecc): ?>
        <div class="col-md-4 mb-4">
            <div class="card h-100" style="border-color: <?= $ecc['color'] ?>; border-width: 2px;">
                <div class="card-header" style="background-color: <?= $ecc['color'] ?>20;">
                    <h5>
                        <i class="fas <?= $ecc['icon'] ?>"></i>
                        <?= htmlspecialchars($ecc['ecc_name']) ?>
                    </h5>
                </div>
                <div class="card-body">
                    <p class="text-muted small">
                        <?= htmlspecialchars($ecc['imaginaire_symbolique']) ?>
                    </p>
                    <div class="d-flex justify-content-between">
                        <span><?= $ecc['nb_cycles'] ?> cycles</span>
                        <span><?= $ecc['nb_lessons'] ?> leçons</span>
                    </div>
                    <div class="progress mt-2">
                        <?php 
                        $percent = ($ecc['nb_cycles'] / 6) * 100; // Objectif : au moins 1 cycle par ECC
                        ?>
                        <div class="progress-bar" role="progressbar" 
                             style="width: <?= min($percent, 100) ?>%; background-color: <?= $ecc['color'] ?>">
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php endforeach; ?>
    </div>
    
    <!-- Graphique de répartition -->
    <div class="card mt-4">
        <div class="card-header">
            <h5>Équilibre du Parcours de Formation</h5>
        </div>
        <div class="card-body">
            <canvas id="eccRadar"></canvas>
        </div>
    </div>
</div>

<script>
// Radar des ECC pour visualiser l'équilibre du parcours
const ctx = document.getElementById('eccRadar').getContext('2d');
const eccData = <?= json_encode($ecc_stats) ?>;

new Chart(ctx, {
    type: 'radar',
    data: {
        labels: eccData.map(e => e.ecc_name),
        datasets: [{
            label: 'Répartition actuelle',
            data: eccData.map(e => e.nb_cycles),
            backgroundColor: 'rgba(54, 162, 235, 0.2)',
            borderColor: 'rgba(54, 162, 235, 1)',
            pointBackgroundColor: eccData.map(e => e.color)
        }, {
            label: 'Équilibre idéal',
            data: [2, 2, 2, 2, 2, 2], // Objectif équilibré
            backgroundColor: 'rgba(255, 99, 132, 0.1)',
            borderColor: 'rgba(255, 99, 132, 0.5)',
            borderDash: [5, 5]
        }]
    },
    options: {
        scales: {
            r: {
                beginAtZero: true,
                max: 4
            }
        }
    }
});
</script>
```

#### 5. **Assistant de Création de FPS**

**Nouveau : `fps_assistant.php`**
```php
<?php
require_once 'functions.php';
include 'header.php';
?>

<div class="container mt-4">
    <h2>Assistant de Création de Forme de Pratique Scolaire</h2>
    
    <form id="fpsWizard" class="mt-4">
        <!-- Étape 1 : Choix PPSAD et ECC -->
        <div class="step" id="step1">
            <h4>Étape 1 : Contexte</h4>
            <div class="mb-3">
                <label>PPSAD</label>
                <select class="form-select" id="ppsad" required>
                    <option value="">Sélectionnez...</option>
                    <option value="Natation">Natation</option>
                    <option value="Handball">Handball</option>
                    <option value="Escalade">Escalade</option>
                    <option value="Danse">Danse</option>
                    <option value="Course en durée">Course en durée</option>
                    <option value="Gymnastique">Gymnastique</option>
                </select>
            </div>
            
            <div class="mb-3">
                <label>Type d'Expérience Corporelle et Culturelle</label>
                <div class="row" id="eccSelector">
                    <!-- Cards ECC générées dynamiquement -->
                </div>
            </div>
        </div>
        
        <!-- Étape 2 : Objet d'Enseignement -->
        <div class="step" id="step2" style="display:none;">
            <h4>Étape 2 : Objet d'Enseignement (Pas en avant)</h4>
            <div class="alert alert-info">
                <i class="fas fa-info-circle"></i>
                L'OE est le "pas en avant" prioritaire que vous visez pour cette séquence.
            </div>
            
            <div id="oePropositions">
                <!-- Suggestions d'OE selon PPSAD et niveau -->
            </div>
            
            <div class="mb-3">
                <label>Votre Objet d'Enseignement</label>
                <textarea class="form-control" id="objetEnseignement" rows="2"></textarea>
            </div>
        </div>
        
        <!-- Étape 3 : Contraintes Emblématiques -->
        <div class="step" id="step3" style="display:none;">
            <h4>Étape 3 : Contraintes Emblématiques</h4>
            <div class="alert alert-warning">
                Les contraintes emblématiques orientent l'activité de l'élève vers l'OE visé.
            </div>
            
            <div class="mb-3">
                <label>Contrainte principale</label>
                <input type="text" class="form-control" id="contraintePrincipale"
                       placeholder="Ex: Jouer uniquement en coup droit">
            </div>
            
            <div class="mb-3">
                <label>Contraintes secondaires</label>
                <textarea class="form-control" id="contraintesSecondaires" rows="3"
                          placeholder="Ex: Terrain réduit, temps limité..."></textarea>
            </div>
        </div>
        
        <!-- Étape 4 : Valeurs et Projet -->
        <div class="step" id="step4" style="display:none;">
            <h4>Étape 4 : Valeurs de l'École</h4>
            
            <div class="mb-3">
                <label>Valeurs promues</label>
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" value="egalite" id="valEgalite">
                    <label class="form-check-label" for="valEgalite">
                        Égalité (filles-garçons, sportifs/non-sportifs)
                    </label>
                </div>
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" value="cooperation" id="valCooperation">
                    <label class="form-check-label" for="valCooperation">
                        Coopération et entraide
                    </label>
                </div>
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" value="autonomie" id="valAutonomie">
                    <label class="form-check-label" for="valAutonomie">
                        Autonomie et responsabilisation
                    </label>
                </div>
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" value="inclusion" id="valInclusion">
                    <label class="form-check-label" for="valInclusion">
                        Inclusion de tous les élèves
                    </label>
                </div>
            </div>
        </div>
        
        <!-- Étape 5 : Synthèse -->
        <div class="step" id="step5" style="display:none;">
            <h4>Étape 5 : Votre FPS</h4>
            
            <div class="card">
                <div class="card-header bg-success text-white">
                    <h5 id="fpsTitre"></h5>
                </div>
                <div class="card-body">
                    <p><strong>PPSAD :</strong> <span id="fpsResumePPSAD"></span></p>
                    <p><strong>ECC :</strong> <span id="fpsResumeECC"></span></p>
                    <p><strong>Objet d'enseignement :</strong> <span id="fpsResumeOE"></span></p>
                    <p><strong>Contraintes emblématiques :</strong> <span id="fpsResumeContraintes"></span></p>
                    <p><strong>Valeurs :</strong> <span id="fpsResumeValeurs"></span></p>
                    
                    <div class="alert alert-success mt-3">
                        <h6>Nom suggéré pour votre FPS :</h6>
                        <input type="text" class="form-control" id="fpsNomFinal" 
                               placeholder="Ex: Le 12 secondes, Tout en CD, La mouli-vache...">
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Navigation -->
        <div class="d-flex justify-content-between mt-4">
            <button type="button" class="btn btn-secondary" id="btnPrevious" style="display:none;">
                Précédent
            </button>
            <button type="button" class="btn btn-primary" id="btnNext">
                Suivant
            </button>
            <button type="submit" class="btn btn-success" id="btnSubmit" style="display:none;">
                Créer la FPS
            </button>
        </div>
    </form>
</div>

<script>
// Wizard de création guidée
let currentStep = 1;
const totalSteps = 5;

function showStep(step) {
    document.querySelectorAll('.step').forEach(s => s.style.display = 'none');
    document.getElementById('step' + step).style.display = 'block';
    
    document.getElementById('btnPrevious').style.display = step > 1 ? 'block' : 'none';
    document.getElementById('btnNext').style.display = step < totalSteps ? 'block' : 'none';
    document.getElementById('btnSubmit').style.display = step === totalSteps ? 'block' : 'none';
}

document.getElementById('btnNext').addEventListener('click', () => {
    if (currentStep < totalSteps) {
        currentStep++;
        showStep(currentStep);
    }
});

document.getElementById('btnPrevious').addEventListener('click', () => {
    if (currentStep > 1) {
        currentStep--;
        showStep(currentStep);
    }
});

// Chargement dynamique des suggestions d'OE
document.getElementById('ppsad').addEventListener('change', function() {
    fetch('api/get_oe_suggestions.php?ppsad=' + this.value)
        .then(response => response.json())
        .then(data => {
            // Afficher les suggestions d'OE
        });
});
</script>
```

---

### Phase 3 : Outils Pédagogiques Avancés

#### 6. **Générateur de Séquences Complètes**

```php
<?php
// generate_sequence.php
class SequenceGenerator {
    private $pdo;
    private $cedrepsLibrary;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
        $this->loadCEDREPSLibrary();
    }
    
    private function loadCEDREPSLibrary() {
        // Charger les références des cahiers CEDREPS
        $this->cedrepsLibrary = [
            'natation' => [
                'cycle3' => [
                    'oe' => 'De la fréquence vers l\'amplitude',
                    'fps' => 'Le 12 secondes',
                    'contraintes' => ['Nager 12 secondes', 'Compter ses coups de bras'],
                    'source' => 'Cahier 15, Colombet & Pezelier'
                ],
                'cycle4' => [
                    'oe' => 'Champion de soi-même',
                    'fps' => 'Autoréférence en crawl',
                    'contraintes' => ['3 essais pour battre son record', 'Choix de la distance'],
                    'source' => 'Cahier 15, Llobet'
                ]
            ],
            // ... autres PPSAD
        ];
    }
    
    public function generateSequence($ppsad, $niveau, $nbLessons = 10) {
        $sequence = [];
        $ref = $this->cedrepsLibrary[$ppsad][$niveau] ?? null;
        
        if (!$ref) {
            return null;
        }
        
        // Phase 1 : Découverte (2 leçons)
        $sequence[] = [
            'numero' => 1,
            'titre' => 'Découverte de la FPS',
            'fps' => $ref['fps'],
            'cp' => 'Comprendre les règles et contraintes',
            'evaluation_type' => 'diagnostique'
        ];
        
        // Phase 2 : Apprentissage (6 leçons) - Double boucle
        for ($i = 2; $i <= 7; $i++) {
            $sequence[] = [
                'numero' => $i,
                'titre' => ($i % 2 == 0) ? 'Grande boucle - Pratique globale' : 'Petite boucle - Travail spécifique',
                'fps' => $ref['fps'],
                'cp' => $this->getProgressiveCP($i, $ref['oe']),
                'evaluation_type' => 'formative'
            ];
        }
        
        // Phase 3 : Consolidation (2 leçons)
        $sequence[] = [
            'numero' => 8,
            'titre' => 'Consolidation et régulation',
            'fps' => $ref['fps'] . ' - Variantes',
            'cp' => 'Adapter sa stratégie',
            'evaluation_type' => 'formative'
        ];
        
        // Phase 4 : Évaluation finale
        $sequence[] = [
            'numero' => 9,
            'titre' => 'Évaluation sommative',
            'fps' => $ref['fps'],
            'cp' => 'Mobiliser l\'ensemble des acquis',
            'evaluation_type' => 'sommative'
        ];
        
        $sequence[] = [
            'numero' => 10,
            'titre' => 'Bilan et perspectives',
            'fps' => 'Situations au choix',
            'cp' => 'Autoévaluation et projet personnel',
            'evaluation_type' => 'bilan'
        ];
        
        return $sequence;
    }
    
    private function getProgressiveCP($lessonNum, $oe) {
        // Contenus prioritaires progressifs selon l'OE
        $progressions = [
            2 => 'Repères sur soi',
            3 => 'Ajustement des paramètres',
            4 => 'Sensations et ressentis',
            5 => 'Coordination et fluidité',
            6 => 'Efficacité et rendement',
            7 => 'Stratégies personnalisées'
        ];
        
        return $progressions[$lessonNum] ?? 'Consolidation';
    }
}
?>
```

#### 7. **Analyseur de Cohérence Pédagogique**

```php
<?php
// coherence_analyzer.php
class CoherenceAnalyzer {
    private $pdo;
    
    public function analyzeCycle($cycleId) {
        $analysis = [
            'score' => 0,
            'strengths' => [],
            'weaknesses' => [],
            'recommendations' => []
        ];
        
        // Récupérer les données du cycle
        $stmt = $this->pdo->prepare('
            SELECT c.*, COUNT(l.id) as nb_lessons,
                   AVG(e.score) as avg_score
            FROM cycles c
            LEFT JOIN lessons l ON c.id = l.cycle_id
            LEFT JOIN evaluations e ON l.id = e.lesson_id
            WHERE c.id = ?
            GROUP BY c.id
        ');
        $stmt->execute([$cycleId]);
        $cycle = $stmt->fetch();
        
        // Critère 1 : Présence des éléments CEDREPS
        if (!empty($cycle['ppsad'])) {
            $analysis['score'] += 20;
            $analysis['strengths'][] = 'PPSAD définie';
        } else {
            $analysis['weaknesses'][] = 'PPSAD non définie';
            $analysis['recommendations'][] = 'Préciser la PPSAD du cycle';
        }
        
        if (!empty($cycle['objective'])) {
            $analysis['score'] += 20;
            $analysis['strengths'][] = 'Objet d\'enseignement ciblé';
        } else {
            $analysis['weaknesses'][] = 'Objet d\'enseignement absent';
            $analysis['recommendations'][] = 'Définir un OE clair (pas en avant)';
        }
        
        if (!empty($cycle['ecc_type'])) {
            $analysis['score'] += 15;
            $analysis['strengths'][] = 'Type d\'ECC identifié';
        }
        
        // Critère 2 : Nombre de leçons
        if ($cycle['nb_lessons'] >= 8) {
            $analysis['score'] += 15;
            $analysis['strengths'][] = 'Durée suffisante (' . $cycle['nb_lessons'] . ' leçons)';
        } else {
            $analysis['weaknesses'][] = 'Séquence trop courte';
            $analysis['recommendations'][] = 'Prévoir au moins 8 leçons pour permettre les transformations';
        }
        
        // Critère 3 : Progression des évaluations
        if ($cycle['avg_score'] !== null) {
            $stmt = $this->pdo->prepare('
                SELECT l.lesson_date, AVG(e.score) as lesson_avg
                FROM lessons l
                LEFT JOIN evaluations e ON l.id = e.lesson_id
                WHERE l.cycle_id = ?
                GROUP BY l.id
                ORDER BY l.lesson_date
            ');
            $stmt->execute([$cycleId]);
            $progression = $stmt->fetchAll();
            
            // Calculer la tendance
            if (count($progression) > 1) {
                $first_avg = $progression[0]['lesson_avg'];
                $last_avg = $progression[count($progression) - 1]['lesson_avg'];
                
                if ($last_avg > $first_avg) {
                    $analysis['score'] += 20;
                    $analysis['strengths'][] = 'Progression positive observée';
                } else {
                    $analysis['weaknesses'][] = 'Pas de progression claire';
                    $analysis['recommendations'][] = 'Analyser les obstacles à l\'apprentissage';
                }
            }
        }
        
        // Critère 4 : Utilisation des 4 dimensions
        $stmt = $this->pdo->prepare('
            SELECT 
                COUNT(DISTINCT CASE WHEN motricite IS NOT NULL THEN student_id END) as eval_motricite,
                COUNT(DISTINCT CASE WHEN intelligibilite IS NOT NULL THEN student_id END) as eval_intelligibilite,
                COUNT(DISTINCT CASE WHEN sensibilite IS NOT NULL THEN student_id END) as eval_sensibilite,
                COUNT(DISTINCT CASE WHEN sociabilite IS NOT NULL THEN student_id END) as eval_sociabilite
            FROM evaluations e
            JOIN lessons l ON e.lesson_id = l.id
            WHERE l.cycle_id = ?
        ');
        $stmt->execute([$cycleId]);
        $dimensions = $stmt->fetch();
        
        $dimensions_used = 0;
        foreach (['motricite', 'intelligibilite', 'sensibilite', 'sociabilite'] as $dim) {
            if ($dimensions['eval_' . $dim] > 0) {
                $dimensions_used++;
            }
        }
        
        if ($dimensions_used == 4) {
            $analysis['score'] += 10;
            $analysis['strengths'][] = 'Les 4 dimensions du pratiquant cultivé sont évaluées';
        } else {
            $analysis['weaknesses'][] = 'Seulement ' . $dimensions_used . '/4 dimensions évaluées';
            $analysis['recommendations'][] = 'Évaluer toutes les dimensions (motricité, intelligibilité, sensibilité, sociabilité)';
        }
        
        // Niveau de cohérence
        if ($analysis['score'] >= 80) {
            $analysis['level'] = 'Excellent';
            $analysis['color'] = 'success';
        } elseif ($analysis['score'] >= 60) {
            $analysis['level'] = 'Bon';
            $analysis['color'] = 'info';
        } elseif ($analysis['score'] >= 40) {
            $analysis['level'] = 'À améliorer';
            $analysis['color'] = 'warning';
        } else {
            $analysis['level'] = 'Insuffisant';
            $analysis['color'] = 'danger';
        }
        
        return $analysis;
    }
}
?>
```

---

### Phase 4 : Partage et Collaboration

#### 8. **Bibliothèque de FPS Partagées**

```sql
-- Table pour le partage de FPS entre enseignants
CREATE TABLE IF NOT EXISTS fps_library (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    ppsad VARCHAR(100) NOT NULL,
    ecc_type VARCHAR(10),
    niveau_scolaire ENUM('cycle3', 'cycle4', 'lycee') NOT NULL,
    objet_enseignement TEXT,
    contraintes_emblematiques TEXT,
    description TEXT,
    fps_details TEXT,
    cp_suggestions TEXT,
    author_id INT,
    etablissement VARCHAR(255),
    academie VARCHAR(100),
    validated_cedreps BOOLEAN DEFAULT FALSE,
    likes_count INT DEFAULT 0,
    downloads_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (author_id) REFERENCES users(id),
    INDEX idx_ppsad_niveau (ppsad, niveau_scolaire),
    INDEX idx_validated (validated_cedreps)
);

-- Table des téléchargements/likes
CREATE TABLE IF NOT EXISTS fps_interactions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    fps_id INT NOT NULL,
    user_id INT NOT NULL,
    interaction_type ENUM('like', 'download', 'adapt') NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (fps_id) REFERENCES fps_library(id),
    FOREIGN KEY (user_id) REFERENCES users(id),
    UNIQUE KEY unique_like (fps_id, user_id, interaction_type)
);
```

#### 9. **Export/Import au Format CEDREPS**

```php
<?php
// export_cedreps_format.php
class CEDREPSExporter {
    
    public function exportCycle($cycleId, $format = 'json') {
        $data = [
            'version' => '2.0',
            'framework' => 'CEDREPS',
            'exported_at' => date('Y-m-d H:i:s'),
            'cycle' => $this->getCycleData($cycleId),
            'lessons' => $this->getLessonsData($cycleId),
            'evaluations' => $this->getEvaluationsData($cycleId)
        ];
        
        if ($format === 'json') {
            return json_encode($data, JSON_PRETTY_PRINT);
        } elseif ($format === 'xml') {
            return $this->arrayToXml($data);
        } elseif ($format === 'cedreps') {
            return $this->toCEDREPSFormat($data);
        }
    }
    
    private function toCEDREPSFormat($data) {
        // Format spécifique compatible avec les cahiers CEDREPS
        $output = "=== FICHE FPS - FORMAT CEDREPS ===\n\n";
        
        $output .= "IDENTIFICATION\n";
        $output .= "--------------\n";
        $output .= "PPSAD : " . $data['cycle']['ppsad'] . "\n";
        $output .= "Type ECC : " . $data['cycle']['ecc_type'] . "\n";
        $output .= "Niveau : " . $data['cycle']['niveau'] . "\n\n";
        
        $output .= "OBJET D'ENSEIGNEMENT\n";
        $output .= "--------------------\n";
        $output .= $data['cycle']['objective'] . "\n\n";
        
        $output .= "FORME DE PRATIQUE SCOLAIRE\n";
        $output .= "--------------------------\n";
        $output .= "Nom : " . $data['cycle']['fps_name'] . "\n";
        $output .= "Contraintes emblématiques :\n";
        foreach ($data['cycle']['contraintes'] as $contrainte) {
            $output .= "- " . $contrainte . "\n";
        }
        $output .= "\n";
        
        $output .= "FONDS CULTUREL\n";
        $output .= "--------------\n";
        $output .= "Mobiles : " . $data['cycle']['fonds_culturel_mobiles'] . "\n";
        $output .= "Principes : " . $data['cycle']['fonds_culturel_principes'] . "\n";
        $output .= "Rapport à l'autre : " . $data['cycle']['fonds_culturel_rapport_autre'] . "\n\n";
        
        $output .= "PROGRESSION (Double boucle)\n";
        $output .= "--------------------------\n";
        foreach ($data['lessons'] as $i => $lesson) {
            $output .= "Leçon " . ($i + 1) . " : " . $lesson['title'] . "\n";
            $output .= "  FPS : " . $lesson['fps'] . "\n";
            $output .= "  CP : " . $lesson['cp'] . "\n";
        }
        
        return $output;
    }
    
    public function importCEDREPSFormat($content, $userId) {
        // Parser le format CEDREPS et créer les entrées en base
        $parsed = $this->parseCEDREPSFormat($content);
        
        // Créer le cycle
        $cycleId = $this->createCycleFromImport($parsed, $userId);
        
        // Créer les leçons
        foreach ($parsed['lessons'] as $lesson) {
            $this->createLessonFromImport($lesson, $cycleId);
        }
        
        return $cycleId;
    }
}
?>
```

---

### Phase 5 : Formation et Accompagnement

#### 10. **Module de Formation Intégré**

```php
<!-- formation_cedreps.php -->
<div class="container mt-4">
    <h2>Formation CEDREPS Intégrée</h2>
    
    <!-- Parcours de formation -->
    <div class="row mt-4">
        <div class="col-md-3">
            <div class="list-group">
                <a href="#module1" class="list-group-item list-group-item-action active">
                    <i class="fas fa-book"></i> Module 1 : Les concepts fondamentaux
                </a>
                <a href="#module2" class="list-group-item list-group-item-action">
                    <i class="fas fa-compass"></i> Module 2 : Les 6 ECC
                </a>
                <a href="#module3" class="list-group-item list-group-item-action">
                    <i class="fas fa-bullseye"></i> Module 3 : Cibler un OE
                </a>
                <a href="#module4" class="list-group-item list-group-item-action">
                    <i class="fas fa-cogs"></i> Module 4 : Construire une FPS
                </a>
                <a href="#module5" class="list-group-item list-group-item-action">
                    <i class="fas fa-sync"></i> Module 5 : La double boucle
                </a>
                <a href="#module6" class="list-group-item list-group-item-action">
                    <i class="fas fa-chart-line"></i> Module 6 : Évaluer les 4 dimensions
                </a>
            </div>
        </div>
        
        <div class="col-md-9">
            <div id="module1" class="module-content">
                <h3>Module 1 : Les Concepts Fondamentaux du CEDREPS</h3>
                
                <div class="accordion" id="conceptsAccordion">
                    <!-- PPSAD -->
                    <div class="accordion-item">
                        <h2 class="accordion-header">
                            <button class="accordion-button" type="button" data-bs-toggle="collapse" data-bs-target="#concept1">
                                Qu'est-ce qu'une PPSAD ?
                            </button>
                        </h2>
                        <div id="concept1" class="accordion-collapse collapse show">
                            <div class="accordion-body">
                                <p><strong>Pratique Physique Sportive Artistique et de Développement</strong></p>
                                <p>Evolution du terme APSA, la PPSAD inclut:</p>
                                <ul>
                                    <li>Les pratiques sportives traditionnelles</li>
                                    <li>Les pratiques artistiques (danse, cirque)</li>
                                    <li>Les pratiques de développement personnel (yoga, musculation)</li>
                                </ul>
                                <div class="alert alert-info mt-3">
                                    <i class="fas fa-lightbulb"></i>
                                    <strong>Point clé :</strong> Une même PPSAD peut offrir plusieurs expériences corporelles 
                                    (ex: escalade de vitesse ≠ escalade en grande voie)
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- FPS -->
                    <div class="accordion-item">
                        <h2 class="accordion-header">
                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#concept2">
                                La Forme de Pratique Scolaire (FPS)
                            </button>
                        </h2>
                        <div id="concept2" class="accordion-collapse collapse">
                            <div class="accordion-body">
                                <p>Une FPS est une <strong>création originale</strong> de l'enseignant qui:</p>
                                <ol>
                                    <li>Encapsule un objet d'enseignement précis</li>
                                    <li>Propose des contraintes emblématiques</li>
                                    <li>Reste valide sur toute une séquence (15h minimum)</li>
                                    <li>S'adapte aux ressources des élèves</li>
                                </ol>
                                
                                <div class="card mt-3">
                                    <div class="card-header bg-primary text-white">
                                        <h6>Exemple : "Le 12 secondes" en natation</h6>
                                    </div>
                                    <div class="card-body">
                                        <p><strong>OE :</strong> Passer de la fréquence à l'amplitude</p>
                                        <p><strong>Contrainte :</strong> Nager exactement 12 secondes</p>
                                        <p><strong>But :</strong> Parcourir la plus grande distance en 12s</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Pratiquant Cultivé -->
                    <div class="accordion-item">
                        <h2 class="accordion-header">
                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#concept3">
                                Le Pratiquant Cultivé
                            </button>
                        </h2>
                        <div id="concept3" class="accordion-collapse collapse">
                            <div class="accordion-body">
                                <p>Former un pratiquant cultivé, c'est développer <strong>4 dimensions</strong>:</p>
                                
                                <div class="row mt-3">
                                    <div class="col-md-6">
                                        <div class="card border-primary mb-3">
                                            <div class="card-header">🏃 Motricité</div>
                                            <div class="card-body">
                                                <p>Le "corps vital", les transformations motrices, le "pas en avant"</p>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="card border-info mb-3">
                                            <div class="card-header">🧠 Intelligibilité</div>
                                            <div class="card-body">
                                                <p>Compréhension, analyse, construction de stratégies</p>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="card border-warning mb-3">
                                            <div class="card-header">❤️ Sensibilité</div>
                                            <div class="card-body">
                                                <p>Ressentis, émotions, rapport sensible au corps et à l'activité</p>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="card border-success mb-3">
                                            <div class="card-header">👥 Sociabilité</div>
                                            <div class="card-body">
                                                <p>Rapport aux autres, coopération, communication, rôles sociaux</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Quiz interactif -->
                <div class="card mt-4">
                    <div class="card-header bg-secondary text-white">
                        <h5>🎯 Auto-évaluation Module 1</h5>
                    </div>
                    <div class="card-body">
                        <form id="quiz1">
                            <div class="mb-3">
                                <p><strong>1. Une FPS est avant tout:</strong></p>
                                <div class="form-check">
                                    <input class="form-check-input" type="radio" name="q1" value="a">
                                    <label class="form-check-label">Une simplification de l'APSA</label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="radio" name="q1" value="b">
                                    <label class="form-check-label">Une création originale encapsulant un OE</label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="radio" name="q1" value="c">
                                    <label class="form-check-label">Un exercice technique</label>
                                </div>
                            </div>
                            
                            <button type="submit" class="btn btn-primary">Vérifier</button>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
```

---

## 📊 Tableau de Bord de Conformité CEDREPS

```php
<!-- dashboard_conformite.php -->
<?php
$stats = [
    'cycles_avec_oe' => $pdo->query("SELECT COUNT(*) FROM cycles WHERE objective IS NOT NULL")->fetchColumn(),
    'cycles_total' => $pdo->query("SELECT COUNT(*) FROM cycles")->fetchColumn(),
    'lessons_avec_fps' => $pdo->query("SELECT COUNT(*) FROM lessons WHERE fps IS NOT NULL")->fetchColumn(),
    'lessons_total' => $pdo->query("SELECT COUNT(*) FROM lessons")->fetchColumn(),
    'ecc_couvertes' => $pdo->query("SELECT COUNT(DISTINCT ecc_type) FROM cycles WHERE ecc_type IS NOT NULL")->fetchColumn(),
    'eval_4d' => $pdo->query("SELECT COUNT(*) FROM evaluations WHERE motricite IS NOT NULL AND intelligibilite IS NOT NULL AND sensibilite IS NOT NULL AND sociabilite IS NOT NULL")->fetchColumn(),
    'eval_total' => $pdo->query("SELECT COUNT(*) FROM evaluations")->fetchColumn()
];

$conformite_score = 0;
$conformite_score += ($stats['cycles_avec_oe'] / max($stats['cycles_total'], 1)) * 25;
$conformite_score += ($stats['lessons_avec_fps'] / max($stats['lessons_total'], 1)) * 25;
$conformite_score += ($stats['ecc_couvertes'] / 6) * 25;
$conformite_score += ($stats['eval_4d'] / max($stats['eval_total'], 1)) * 25;
?>

<div class="container mt-4">
    <h2>Tableau de Bord - Conformité CEDREPS</h2>
    
    <div class="row mt-4">
        <div class="col-md-4">
            <div class="card">
                <div class="card-body text-center">
                    <h1 class="display-3 <?= $conformite_score >= 75 ? 'text-success' : ($conformite_score >= 50 ? 'text-warning' : 'text-danger') ?>">
                        <?= round($conformite_score) ?>%
                    </h1>
                    <p>Score de Conformité Global</p>
                </div>
            </div>
        </div>
        
        <div class="col-md-8">
            <div class="card">
                <div class="card-body">
                    <h5>Indicateurs Détaillés</h5>
                    
                    <div class="mb-3">
                        <label>Cycles avec Objet d'Enseignement</label>
                        <div class="progress">
                            <div class="progress-bar" style="width: <?= ($stats['cycles_avec_oe'] / max($stats['cycles_total'], 1)) * 100 ?>%">
                                <?= $stats['cycles_avec_oe'] ?>/<?= $stats['cycles_total'] ?>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label>Leçons avec FPS définies</label>
                        <div class="progress">
                            <div class="progress-bar bg-info" style="width: <?= ($stats['lessons_avec_fps'] / max($stats['lessons_total'], 1)) * 100 ?>%">
                                <?= $stats['lessons_avec_fps'] ?>/<?= $stats['lessons_total'] ?>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label>Couverture des 6 ECC</label>
                        <div class="progress">
                            <div class="progress-bar bg-warning" style="width: <?= ($stats['ecc_couvertes'] / 6) * 100 ?>%">
                                <?= $stats['ecc_couvertes'] ?>/6
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label>Évaluations 4 dimensions</label>
                        <div class="progress">
                            <div class="progress-bar bg-success" style="width: <?= ($stats['eval_4d'] / max($stats['eval_total'], 1)) * 100 ?>%">
                                <?= $stats['eval_4d'] ?>/<?= $stats['eval_total'] ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Recommandations -->
    <?php if ($conformite_score < 75): ?>
    <div class="alert alert-info mt-4">
        <h5>📈 Recommandations pour améliorer votre conformité CEDREPS</h5>
        <ul>
            <?php if ($stats['cycles_avec_oe'] < $stats['cycles_total']): ?>
            <li>Définir des Objets d'Enseignement pour tous vos cycles</li>
            <?php endif; ?>
            <?php if ($stats['ecc_couvertes'] < 6): ?>
            <li>Diversifier les Expériences Corporelles et Culturelles (<?= 6 - $stats['ecc_couvertes'] ?> ECC manquantes)</li>
            <?php endif; ?>
            <?php if ($stats['eval_4d'] < $stats['eval_total'] * 0.5): ?>
            <li>Évaluer systématiquement les 4 dimensions du pratiquant cultivé</li>
            <?php endif; ?>
        </ul>
    </div>
    <?php endif; ?>
</div>
```

---

## 🚀 Bénéfices de cette Intégration

### Pour les Enseignants
- **Guidance pédagogique** claire basée sur 30 ans de recherche CEDREPS
- **Outils pratiques** pour créer des FPS pertinentes
- **Formation continue** intégrée à l'outil
- **Partage** et mutualisation des pratiques

### Pour les Élèves
- **Parcours cohérent** à travers les 6 types d'expériences
- **Transformations durables** grâce aux OE ciblés
- **Évaluation complète** des 4 dimensions
- **Engagement** dans des pratiques culturellement signifiantes

### Pour l'Institution
- **Harmonisation** des pratiques autour d'un cadre théorique solide
- **Formation** des nouveaux enseignants facilitée
- **Traçabilité** des apprentissages
- **Innovation pédagogique** documentée

---

## 📅 Planning de Mise en Œuvre

| Phase | Durée | Actions |
|-------|-------|---------|
| **Phase 1** | 1 semaine | Enrichissement base de données + Tables ECC |
| **Phase 2** | 2 semaines | Interface utilisateur enrichie + Dashboard ECC |
| **Phase 3** | 2 semaines | Outils pédagogiques (Assistant FPS, Analyseur) |
| **Phase 4** | 1 semaine | Système de partage et bibliothèque |
| **Phase 5** | Continue | Formation et accompagnement |

---

## 💡 Conclusion

Cette intégration approfondie des concepts CEDREPS transformera votre application en :
- **Outil de référence** pour l'enseignement selon l'approche CEDREPS
- **Plateforme de formation** pour les enseignants
- **Communauté de pratique** pour le partage de FPS
- **Laboratoire d'innovation** pédagogique en EPS

Avec ces enrichissements, votre application deviendra un **pont essentiel** entre la théorie CEDREPS et la pratique quotidienne des enseignants d'EPS.

---

*"La référence, non la révérence" - Principe CEDREPS appliqué à votre outil numérique*