<?php
/**
 * ╔══════════════════════════════════════════════════════════════╗
 * ║   DIAGNOSTIC ET MIGRATION ADAPTATIVE - CEDREPS ULTIMATE       ║
 * ║   Détection automatique de la structure des tables            ║
 * ╚══════════════════════════════════════════════════════════════╝
 */

session_start();
$migration_password = 'MigrationCedreps2025';

// Vérification du mot de passe
if (!isset($_GET['pass']) || $_GET['pass'] !== $migration_password) {
    die('Accès refusé - Ajoutez ?pass=MigrationCedreps2025 à l\'URL');
}

$_SESSION['migration_authorized'] = true;

// Configuration des bases
$old_config = [
    'host' => 'localhost',
    'name' => 'epsweb_appli',
    'user' => 'epsweb_appli',
    'pass' => 'MF03444janvierA'
];

// Si requête AJAX pour diagnostic
if (isset($_POST['action']) && $_POST['action'] === 'diagnose') {
    header('Content-Type: application/json');
    performDiagnosis();
    exit;
}

// Si requête AJAX pour migration
if (isset($_POST['action']) && $_POST['action'] === 'migrate') {
    header('Content-Type: application/json');
    performSmartMigration();
    exit;
}

// ===================================================================
// FONCTION DE DIAGNOSTIC
// ===================================================================

function performDiagnosis() {
    global $old_config;
    
    $response = ['status' => 'running', 'tables' => []];
    
    try {
        $pdo = new PDO(
            "mysql:host={$old_config['host']};dbname={$old_config['name']};charset=utf8mb4",
            $old_config['user'],
            $old_config['pass']
        );
        
        // Analyser chaque table
        $tables = ['users', 'classes', 'cycles', 'lessons', 'students', 'evaluations'];
        
        foreach ($tables as $table) {
            try {
                // Récupérer la structure
                $stmt = $pdo->query("SHOW COLUMNS FROM $table");
                $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                $columnNames = [];
                foreach ($columns as $col) {
                    $columnNames[] = $col['Field'];
                }
                
                // Compter les enregistrements
                $count = $pdo->query("SELECT COUNT(*) as total FROM $table")->fetch();
                
                // Récupérer un échantillon
                $sample = $pdo->query("SELECT * FROM $table LIMIT 1")->fetch(PDO::FETCH_ASSOC);
                
                $response['tables'][$table] = [
                    'exists' => true,
                    'columns' => $columnNames,
                    'count' => $count['total'],
                    'sample' => $sample
                ];
                
            } catch (Exception $e) {
                $response['tables'][$table] = [
                    'exists' => false,
                    'error' => $e->getMessage()
                ];
            }
        }
        
        $response['status'] = 'completed';
        
    } catch (Exception $e) {
        $response['status'] = 'error';
        $response['error'] = $e->getMessage();
    }
    
    echo json_encode($response);
}

// ===================================================================
// FONCTION DE MIGRATION INTELLIGENTE
// ===================================================================

function performSmartMigration() {
    global $old_config;
    
    $response = [
        'status' => 'running',
        'messages' => [],
        'progress' => 0,
        'stats' => []
    ];
    
    // Charger la configuration de la nouvelle base
    if (!file_exists(__DIR__ . '/config.php')) {
        $response['status'] = 'error';
        $response['messages'][] = ['type' => 'error', 'text' => 'Fichier config.php manquant'];
        echo json_encode($response);
        return;
    }
    
    require_once __DIR__ . '/config.php';
    $new_config = [
        'host' => $config['database']['host'] ?? 'localhost',
        'name' => $config['database']['name'],
        'user' => $config['database']['user'],
        'pass' => $config['database']['pass']
    ];
    
    try {
        // Connexions
        $pdo_old = new PDO(
            "mysql:host={$old_config['host']};dbname={$old_config['name']};charset=utf8mb4",
            $old_config['user'],
            $old_config['pass']
        );
        
        $pdo_new = new PDO(
            "mysql:host={$new_config['host']};dbname={$new_config['name']};charset=utf8mb4",
            $new_config['user'],
            $new_config['pass']
        );
        
        $response['messages'][] = ['type' => 'success', 'text' => '✅ Connexion aux bases réussie'];
        
        // ===================================================================
        // MIGRATION DES LEÇONS AVEC DÉTECTION AUTOMATIQUE
        // ===================================================================
        
        $response['messages'][] = ['type' => 'header', 'text' => '📖 Migration intelligente des leçons...'];
        
        // Détecter les colonnes de la table lessons
        $stmt = $pdo_old->query("SHOW COLUMNS FROM lessons");
        $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $columnMap = [];
        
        foreach ($columns as $col) {
            $field = strtolower($col['Field']);
            
            // Mapping intelligent des colonnes
            if (strpos($field, 'number') !== false || strpos($field, 'num') !== false || $field === 'numero') {
                $columnMap['lesson_number'] = $col['Field'];
            } elseif (strpos($field, 'cycle') !== false && strpos($field, 'id') !== false) {
                $columnMap['cycle_id'] = $col['Field'];
            } elseif (strpos($field, 'date') !== false) {
                $columnMap['lesson_date'] = $col['Field'];
            } elseif ($field === 'title' || $field === 'titre' || $field === 'name') {
                $columnMap['title'] = $col['Field'];
            } elseif ($field === 'fps') {
                $columnMap['fps'] = $col['Field'];
            } elseif ($field === 'cp' || strpos($field, 'contenu') !== false) {
                $columnMap['cp'] = $col['Field'];
            } elseif (strpos($field, 'desc') !== false || $field === 'content') {
                $columnMap['description'] = $col['Field'];
            } elseif (strpos($field, 'material') !== false || $field === 'materiel') {
                $columnMap['material'] = $col['Field'];
            }
        }
        
        $response['messages'][] = ['type' => 'info', 'text' => '  • Colonnes détectées : ' . json_encode(array_keys($columnMap))];
        
        // Construire la requête dynamiquement
        $selectColumns = [];
        foreach ($columnMap as $newName => $oldName) {
            $selectColumns[] = "`$oldName` as $newName";
        }
        
        // Ajouter les colonnes manquantes avec des valeurs par défaut
        if (!isset($columnMap['lesson_number'])) {
            $selectColumns[] = "id as lesson_number"; // Utiliser l'ID comme numéro si pas de colonne number
        }
        
        $selectQuery = "SELECT " . implode(', ', $selectColumns) . ", id FROM lessons";
        
        // Si on a une colonne pour trier, on l'utilise
        if (isset($columnMap['lesson_number'])) {
            $selectQuery .= " ORDER BY cycle_id, `{$columnMap['lesson_number']}`";
        } else {
            $selectQuery .= " ORDER BY cycle_id, id";
        }
        
        $response['messages'][] = ['type' => 'info', 'text' => '  • Requête construite dynamiquement'];
        
        // Récupérer les leçons
        $lessons = $pdo_old->query($selectQuery)->fetchAll(PDO::FETCH_ASSOC);
        $lessons_migrated = 0;
        $lessons_skipped = 0;
        
        // Récupérer le mapping des cycles (supposé déjà fait)
        $cycle_mapping = [];
        $cycles_result = $pdo_new->query("SELECT id, name FROM cycles")->fetchAll(PDO::FETCH_ASSOC);
        foreach ($cycles_result as $cycle) {
            // Mapper par nom ou par ID
            $cycle_mapping[$cycle['name']] = $cycle['id'];
        }
        
        foreach ($lessons as $lesson) {
            // Trouver le cycle correspondant
            $new_cycle_id = null;
            
            // D'abord essayer de mapper par ID si on a un mapping
            if (isset($lesson['cycle_id'])) {
                // Récupérer le nom du cycle dans l'ancienne base
                $old_cycle = $pdo_old->query("SELECT name FROM cycles WHERE id = {$lesson['cycle_id']}")->fetch();
                if ($old_cycle && isset($cycle_mapping[$old_cycle['name']])) {
                    $new_cycle_id = $cycle_mapping[$old_cycle['name']];
                }
            }
            
            if ($new_cycle_id) {
                // Déterminer le numéro de leçon
                $lesson_number = $lesson['lesson_number'] ?? 1;
                
                // Vérifier si existe
                $check = $pdo_new->prepare("
                    SELECT id FROM lessons 
                    WHERE cycle_id = ? AND lesson_number = ?
                ");
                $check->execute([$new_cycle_id, $lesson_number]);
                
                if (!$check->fetch()) {
                    try {
                        $stmt = $pdo_new->prepare("
                            INSERT INTO lessons (
                                cycle_id, lesson_number, lesson_date, title,
                                fps, cp, description, material, status, created_at
                            ) VALUES (
                                :cycle, :number, :date, :title,
                                :fps, :cp, :desc, :material, :status, NOW()
                            )
                        ");
                        
                        $stmt->execute([
                            ':cycle' => $new_cycle_id,
                            ':number' => $lesson_number,
                            ':date' => $lesson['lesson_date'] ?? date('Y-m-d'),
                            ':title' => $lesson['title'] ?? "Leçon $lesson_number",
                            ':fps' => $lesson['fps'] ?? '',
                            ':cp' => $lesson['cp'] ?? '',
                            ':desc' => $lesson['description'] ?? '',
                            ':material' => $lesson['material'] ?? '',
                            ':status' => 'planned'
                        ]);
                        
                        $lessons_migrated++;
                        
                    } catch (PDOException $e) {
                        // En cas de conflit, essayer avec un nouveau numéro
                        if ($e->getCode() == 23000) {
                            $max = $pdo_new->query("SELECT MAX(lesson_number) as max FROM lessons WHERE cycle_id = $new_cycle_id")->fetch();
                            $new_number = ($max['max'] ?? 0) + 1;
                            
                            $stmt->execute([
                                ':cycle' => $new_cycle_id,
                                ':number' => $new_number,
                                ':date' => $lesson['lesson_date'] ?? date('Y-m-d'),
                                ':title' => $lesson['title'] ?? "Leçon $new_number",
                                ':fps' => $lesson['fps'] ?? '',
                                ':cp' => $lesson['cp'] ?? '',
                                ':desc' => $lesson['description'] ?? '',
                                ':material' => $lesson['material'] ?? '',
                                ':status' => 'planned'
                            ]);
                            
                            $lessons_migrated++;
                            $response['messages'][] = ['type' => 'warning', 'text' => "  ⚠️ Leçon renumérotée: $lesson_number → $new_number"];
                        }
                    }
                } else {
                    $lessons_skipped++;
                }
            } else {
                $lessons_skipped++;
            }
        }
        
        $response['messages'][] = ['type' => 'success', 'text' => "  ✅ $lessons_migrated leçons migrées"];
        if ($lessons_skipped > 0) {
            $response['messages'][] = ['type' => 'warning', 'text' => "  ⚠️ $lessons_skipped leçons ignorées"];
        }
        
        $response['progress'] = 100;
        $response['status'] = 'completed';
        
    } catch (Exception $e) {
        $response['status'] = 'error';
        $response['messages'][] = ['type' => 'error', 'text' => '❌ ERREUR : ' . $e->getMessage()];
    }
    
    echo json_encode($response);
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Diagnostic et Migration - CEDREPS Ultimate</title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
            margin: 0;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
        }
        .card {
            background: white;
            border-radius: 20px;
            padding: 30px;
            margin-bottom: 20px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
        }
        h1 { color: #1F2937; margin-top: 0; }
        h2 { color: #374151; font-size: 20px; margin-top: 30px; }
        .btn {
            display: inline-block;
            padding: 12px 24px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-weight: 600;
            border: none;
            cursor: pointer;
            font-size: 16px;
            margin: 10px 5px;
        }
        .btn:hover { transform: translateY(-2px); }
        .btn.secondary { background: #6B7280; }
        .btn.danger { background: #EF4444; }
        .btn.success { background: #10B981; }
        .table-info {
            margin: 20px 0;
            padding: 15px;
            background: #F9FAFB;
            border-radius: 12px;
            border-left: 4px solid #3B82F6;
        }
        .table-name {
            font-weight: bold;
            color: #1F2937;
            margin-bottom: 10px;
        }
        .columns {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            margin-top: 10px;
        }
        .column-tag {
            background: #E0E7FF;
            color: #3730A3;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 14px;
        }
        .count-badge {
            background: #10B981;
            color: white;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 14px;
            display: inline-block;
        }
        .console {
            background: #1F2937;
            color: #10B981;
            padding: 20px;
            border-radius: 12px;
            font-family: 'Courier New', monospace;
            font-size: 13px;
            line-height: 1.6;
            height: 400px;
            overflow-y: auto;
            margin: 20px 0;
        }
        .loading {
            text-align: center;
            padding: 40px;
        }
        .spinner {
            border: 4px solid #E5E7EB;
            border-top-color: #3B82F6;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
            margin: 20px auto;
        }
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        .alert {
            padding: 15px;
            border-radius: 8px;
            margin: 20px 0;
        }
        .alert-info {
            background: #EFF6FF;
            border: 1px solid #3B82F6;
            color: #1E40AF;
        }
        .alert-success {
            background: #F0FDF4;
            border: 1px solid #10B981;
            color: #166534;
        }
        .alert-warning {
            background: #FFFBEB;
            border: 1px solid #F59E0B;
            color: #92400E;
        }
        .grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin: 20px 0;
        }
        code {
            background: #F3F4F6;
            padding: 2px 6px;
            border-radius: 4px;
            font-family: monospace;
            font-size: 14px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="card">
            <h1>🔍 Diagnostic et Migration Intelligente</h1>
            <p style="color: #6B7280;">Analyse automatique de la structure et migration adaptative</p>
            
            <div class="alert alert-info">
                <strong>ℹ️ Ce script intelligent :</strong>
                <ul style="margin: 10px 0; padding-left: 20px;">
                    <li>Détecte automatiquement les noms de colonnes</li>
                    <li>S'adapte à votre structure de base de données</li>
                    <li>Gère les conflits et les doublons</li>
                    <li>Continue la migration là où elle s'est arrêtée</li>
                </ul>
            </div>
            
            <div style="text-align: center; margin: 30px 0;">
                <button class="btn" onclick="runDiagnosis()">
                    🔍 Analyser la structure
                </button>
                <button class="btn success" onclick="runSmartMigration()" style="display:none;" id="migrate-btn">
                    🚀 Lancer la migration intelligente
                </button>
            </div>
        </div>
        
        <div class="card" id="diagnosis-results" style="display:none;">
            <h2>📊 Résultats du diagnostic</h2>
            <div id="tables-info"></div>
        </div>
        
        <div class="card" id="migration-results" style="display:none;">
            <h2>📦 Migration en cours</h2>
            <div class="console" id="console"></div>
        </div>
    </div>
    
    <script>
        function addConsoleLog(message, type = '') {
            const console = document.getElementById('console');
            const line = document.createElement('div');
            line.style.color = type === 'error' ? '#EF4444' : 
                               type === 'warning' ? '#F59E0B' : 
                               type === 'success' ? '#10B981' : 
                               type === 'info' ? '#3B82F6' : '#10B981';
            line.textContent = message;
            console.appendChild(line);
            console.scrollTop = console.scrollHeight;
        }
        
        async function runDiagnosis() {
            const resultsDiv = document.getElementById('diagnosis-results');
            const tablesInfo = document.getElementById('tables-info');
            
            resultsDiv.style.display = 'block';
            tablesInfo.innerHTML = '<div class="loading"><div class="spinner"></div>Analyse en cours...</div>';
            
            try {
                const response = await fetch(window.location.href, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: 'action=diagnose'
                });
                
                const data = await response.json();
                
                if (data.status === 'completed') {
                    let html = '<div class="grid">';
                    
                    for (const [tableName, tableData] of Object.entries(data.tables)) {
                        if (tableData.exists) {
                            html += `
                                <div class="table-info">
                                    <div class="table-name">
                                        📋 ${tableName} 
                                        <span class="count-badge">${tableData.count} enregistrements</span>
                                    </div>
                                    <div class="columns">
                                        ${tableData.columns.map(col => `<span class="column-tag">${col}</span>`).join('')}
                                    </div>
                                </div>
                            `;
                        }
                    }
                    
                    html += '</div>';
                    
                    // Analyse spécifique pour lessons
                    if (data.tables.lessons && data.tables.lessons.exists) {
                        const lessonCols = data.tables.lessons.columns;
                        html += `
                            <div class="alert alert-warning">
                                <strong>⚠️ Structure de la table lessons :</strong><br>
                                ${lessonCols.includes('lesson_number') ? 
                                    '✅ Colonne lesson_number trouvée' : 
                                    '❌ Colonne lesson_number absente - Utilisation de : ' + 
                                    (lessonCols.find(c => c.includes('number') || c.includes('num')) || 'id')}
                            </div>
                        `;
                    }
                    
                    tablesInfo.innerHTML = html;
                    document.getElementById('migrate-btn').style.display = 'inline-block';
                } else {
                    tablesInfo.innerHTML = '<div class="alert alert-warning">Erreur lors de l\'analyse</div>';
                }
                
            } catch (error) {
                tablesInfo.innerHTML = `<div class="alert alert-warning">Erreur : ${error.message}</div>`;
            }
        }
        
        async function runSmartMigration() {
            const resultsDiv = document.getElementById('migration-results');
            const console = document.getElementById('console');
            
            resultsDiv.style.display = 'block';
            console.innerHTML = '';
            
            addConsoleLog('🚀 Démarrage de la migration intelligente...', 'info');
            addConsoleLog('Cette migration s\'adapte automatiquement à votre structure', 'info');
            
            try {
                const response = await fetch(window.location.href, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: 'action=migrate'
                });
                
                const data = await response.json();
                
                if (data.messages) {
                    data.messages.forEach(msg => {
                        addConsoleLog(msg.text, msg.type);
                    });
                }
                
                if (data.status === 'completed') {
                    addConsoleLog('', '');
                    addConsoleLog('═══════════════════════════════════════', 'success');
                    addConsoleLog('✅ MIGRATION TERMINÉE AVEC SUCCÈS !', 'success');
                    addConsoleLog('═══════════════════════════════════════', 'success');
                    addConsoleLog('', '');
                    addConsoleLog('Vous pouvez maintenant vous connecter à CEDREPS Ultimate', 'info');
                } else if (data.status === 'error') {
                    addConsoleLog('❌ Erreur lors de la migration', 'error');
                }
                
            } catch (error) {
                addConsoleLog(`❌ Erreur : ${error.message}`, 'error');
            }
        }
        
        // Lancer automatiquement le diagnostic au chargement
        window.addEventListener('load', () => {
            setTimeout(runDiagnosis, 500);
        });
    </script>
</body>
</html>