<?php
/**
 * Common functions used across the application
 */

require_once __DIR__ . '/config.php';

session_start();

/**
 * Redirect to a given URL and exit
 *
 * @param string $url
 */
function redirect($url)
{
    header('Location: ' . $url);
    exit;
}

/**
 * Check if a user is logged in
 *
 * @return bool
 */
function isLoggedIn()
{
    return isset($_SESSION['user_id']);
}

/**
 * Retrieve the currently logged in user
 *
 * @param PDO $pdo
 * @return array|null
 */
function currentUser(PDO $pdo)
{
    if (!isLoggedIn()) {
        return null;
    }
    $stmt = $pdo->prepare('SELECT * FROM users WHERE id = ?');
    $stmt->execute([$_SESSION['user_id']]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Log in a user with email and password
 *
 * @param PDO    $pdo
 * @param string $email
 * @param string $password
 * @return bool true on success, false otherwise
 */
function loginUser(PDO $pdo, $email, $password)
{
    $stmt = $pdo->prepare('SELECT * FROM users WHERE email = ?');
    $stmt->execute([$email]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($user && password_verify($password, $user['password'])) {
        $_SESSION['user_id'] = $user['id'];
        return true;
    }
    return false;
}

/**
 * Register a new user (teacher by default)
 *
 * @param PDO    $pdo
 * @param string $name
 * @param string $email
 * @param string $password
 * @param string $role
 * @return bool true on success, false otherwise
 */
function registerUser(PDO $pdo, $name, $email, $password, $role = 'teacher')
{
    // Check if email already exists
    $stmt = $pdo->prepare('SELECT id FROM users WHERE email = ?');
    $stmt->execute([$email]);
    if ($stmt->fetch()) {
        return false;
    }
    $hash = password_hash($password, PASSWORD_DEFAULT);
    $stmt = $pdo->prepare('INSERT INTO users (name, email, password, role) VALUES (?, ?, ?, ?)');
    return $stmt->execute([$name, $email, $hash, $role]);
}

/**
 * Log out the current user
 */
function logoutUser()
{
    session_unset();
    session_destroy();
}

/**
 * Get all classes for a teacher
 *
 * @param PDO $pdo
 * @param int $teacherId
 * @return array
 */
function getClasses(PDO $pdo, $teacherId)
{
    $stmt = $pdo->prepare('SELECT * FROM classes WHERE teacher_id = ? ORDER BY created_at DESC');
    $stmt->execute([$teacherId]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get class details
 *
 * @param PDO $pdo
 * @param int $classId
 * @return array|null
 */
function getClassById(PDO $pdo, $classId)
{
    $stmt = $pdo->prepare('SELECT * FROM classes WHERE id = ?');
    $stmt->execute([$classId]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Get cycles for a class
 *
 * @param PDO $pdo
 * @param int $classId
 * @return array
 */
function getCycles(PDO $pdo, $classId)
{
    $stmt = $pdo->prepare('SELECT * FROM cycles WHERE class_id = ? ORDER BY created_at DESC');
    $stmt->execute([$classId]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get a cycle by ID
 *
 * @param PDO $pdo
 * @param int $cycleId
 * @return array|null
 */
function getCycleById(PDO $pdo, $cycleId)
{
    $stmt = $pdo->prepare('SELECT * FROM cycles WHERE id = ?');
    $stmt->execute([$cycleId]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Get lessons for a cycle
 *
 * @param PDO $pdo
 * @param int $cycleId
 * @return array
 */
function getLessons(PDO $pdo, $cycleId)
{
    $stmt = $pdo->prepare('SELECT * FROM lessons WHERE cycle_id = ? ORDER BY lesson_date ASC');
    $stmt->execute([$cycleId]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get lesson by ID
 *
 * @param PDO $pdo
 * @param int $lessonId
 * @return array|null
 */
function getLessonById(PDO $pdo, $lessonId)
{
    $stmt = $pdo->prepare('SELECT * FROM lessons WHERE id = ?');
    $stmt->execute([$lessonId]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Get students of a class
 *
 * @param PDO $pdo
 * @param int $classId
 * @return array
 */
function getStudents(PDO $pdo, $classId)
{
    $stmt = $pdo->prepare('SELECT * FROM students WHERE class_id = ? ORDER BY name ASC');
    $stmt->execute([$classId]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get evaluations for a lesson
 *
 * @param PDO $pdo
 * @param int $lessonId
 * @return array
 */
function getEvaluations(PDO $pdo, $lessonId)
{
    $stmt = $pdo->prepare(
        'SELECT e.*, s.name AS student_name FROM evaluations e
         JOIN students s ON e.student_id = s.id
         WHERE e.lesson_id = ? ORDER BY s.name ASC'
    );
    $stmt->execute([$lessonId]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get all roles (social roles) defined in the system.
 *
 * @param PDO $pdo
 * @return array
 */
function getRoles(PDO $pdo)
{
    $stmt = $pdo->query('SELECT * FROM roles ORDER BY name ASC');
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get role assignments for a lesson.
 *
 * @param PDO $pdo
 * @param int $lessonId
 * @return array keyed by student_id
 */
function getLessonRoles(PDO $pdo, $lessonId)
{
    $stmt = $pdo->prepare(
        'SELECT lr.id, lr.student_id, lr.role_id, r.name AS role_name FROM lesson_roles lr
         JOIN roles r ON lr.role_id = r.id
         WHERE lr.lesson_id = ?'
    );
    $stmt->execute([$lessonId]);
    $roles = [];
    foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
        $roles[$row['student_id']] = $row;
    }
    return $roles;
}

/**
 * Assign roles to students for a lesson.  This function will
 * overwrite existing assignments.
 *
 * @param PDO   $pdo
 * @param int   $lessonId
 * @param array $assignments associative array student_id => role_id
 */
function setLessonRoles(PDO $pdo, $lessonId, array $assignments)
{
    // Remove previous assignments
    $stmt = $pdo->prepare('DELETE FROM lesson_roles WHERE lesson_id = ?');
    $stmt->execute([$lessonId]);
    // Insert new assignments
    $stmtIns = $pdo->prepare('INSERT INTO lesson_roles (lesson_id, student_id, role_id) VALUES (?, ?, ?)');
    foreach ($assignments as $studentId => $roleId) {
        if ($roleId) {
            $stmtIns->execute([$lessonId, $studentId, $roleId]);
        }
    }
}

/**
 * Get all objects d'enseignement from the library.
 *
 * @param PDO $pdo
 * @return array
 */
function getOELibrary(PDO $pdo)
{
    $stmt = $pdo->query('SELECT * FROM oe_library ORDER BY ppsad, name');
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get a single OE from the library by ID.
 *
 * @param PDO $pdo
 * @param int $oeId
 * @return array|null
 */
function getOEById(PDO $pdo, $oeId)
{
    $stmt = $pdo->prepare('SELECT * FROM oe_library WHERE id = ?');
    $stmt->execute([$oeId]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Get all forms of practice from the FPS library.
 *
 * @param PDO $pdo
 * @return array
 */
function getFPSLibrary(PDO $pdo)
{
    $stmt = $pdo->query('SELECT * FROM fps_library ORDER BY ppsad, name');
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get a single FPS entry from the library by ID.
 *
 * @param PDO $pdo
 * @param int $fpsId
 * @return array|null
 */
function getFPSById(PDO $pdo, $fpsId)
{
    $stmt = $pdo->prepare('SELECT * FROM fps_library WHERE id = ?');
    $stmt->execute([$fpsId]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}
